<?php

/**
  * Admin panel functions, functions.inc.php
  * @category admin
  *
  * @author PrestaShop <support@prestashop.com>
  * @copyright PrestaShop
  * @license http://www.opensource.org/licenses/osl-3.0.php Open-source licence 3.0
  * @version 0.8
  *
  */

/**
  * Generate a cached thumbnail for object lists (eg. carrier, order states...etc)
  *
  * @param string $image Real image filename
  * @param string $cacheImage Cached filename
  * @param integer $size Desired size
  */
function cacheImage($image, $cacheImage, $size, $imageType = 'jpg')
{
	if (file_exists($image))
	{
		if (!file_exists(_PS_TMP_IMG_DIR_.$cacheImage))
		{
			$imageGd = ($imageType == 'gif' ? imagecreatefromgif($image) : imagecreatefromjpeg($image));
			$x = imagesx($imageGd);
			$y = imagesy($imageGd);
			
			/* Size is already ok */
			if ($x < $size) 
				copy($image, _PS_TMP_IMG_DIR_.$cacheImage);

			/* We need to resize */
			else
			{
				$ratioY = $y / ($x / $size);
				$newImage = ($imageType == 'gif' ? imagecreatetruecolor($size, $ratioY) : imagecreatetruecolor($size, $ratioY));
				
				/* Allow to keep nice look even if resized */
				$white = imagecolorallocate($newImage, 255, 255, 255);
				imagefill($newImage, 0, 0, $white);
				imagecopyresampled($newImage, $imageGd, 0, 0, 0, 0, $size, $ratioY, $x, $y);
				imagecolortransparent($newImage, $white);

				/* Quality alteration and image creation */
				if ($imageType == 'gif')
					imagegif($newImage, _PS_TMP_IMG_DIR_.$cacheImage);
				else
					imagejpeg($newImage, _PS_TMP_IMG_DIR_.$cacheImage, 86);
			}
		}
		return '<img src="../img/tmp/'.$cacheImage.'" alt="logo" title="logo" />';
	}
	return '';
}

/**
  * Generate a new settings file, only transmitted parameters are updated
  *
  * @param string $baseUri Base URI
  * @param string $theme Theme name (eg. default)
  * @param array $arrayDB Parameters in order to connect to database
  */
function	rewriteSettingsFile($baseUri = null, $theme = null, $arrayDB = null)
{
 	$defines = array();
	$defines['__PS_BASE_URI__'] = !is_null($baseUri) ? $baseUri : __PS_BASE_URI__;
	$defines['_THEME_NAME_'] = $theme ? $theme : _THEME_NAME_;
	$defines['_DB_NAME_'] = (($arrayDB AND isset($arrayDB['_DB_NAME_'])) ? $arrayDB['_DB_NAME_'] : _DB_NAME_);
	$defines['_DB_SERVER_'] = (($arrayDB AND isset($arrayDB['_DB_SERVER_'])) ? $arrayDB['_DB_SERVER_'] : _DB_SERVER_);
	$defines['_DB_USER_'] = (($arrayDB AND isset($arrayDB['_DB_USER_'])) ? $arrayDB['_DB_USER_'] : _DB_USER_);
	$defines['_DB_PREFIX_'] = (($arrayDB AND isset($arrayDB['_DB_PREFIX_'])) ? $arrayDB['_DB_PREFIX_'] : _DB_PREFIX_);
	$defines['_DB_PASSWD_'] = (($arrayDB AND isset($arrayDB['_DB_PASSWD_'])) ? $arrayDB['_DB_PASSWD_'] : _DB_PASSWD_);
	$defines['_DB_TYPE_'] = (($arrayDB AND isset($arrayDB['_DB_TYPE_'])) ? $arrayDB['_DB_TYPE_'] : _DB_TYPE_);
	$defines['_COOKIE_KEY_'] = addslashes(_COOKIE_KEY_);
	$defines['_COOKIE_IV_'] = addslashes(_COOKIE_IV_);
	$defines['_PS_VERSION_'] = addslashes(_PS_VERSION_);
	$content = "<?php\n\n";
	foreach ($defines as $k => $value)
		$content .= 'define(\''.$k.'\', \''.addslashes($value).'\');'."\n";
	$content .= "\n?>";
	if ($fd = fopen(dirname(__FILE__).'/../config/settings.inc.php', 'w'))
	{
		fwrite($fd, $content);
		fclose($fd);
	}
	else
		Tools::displayError('cannot access settings file');
}

/**
  * Display flags in forms for translations
  *
  * @param array $languages All languages available
  * @param integer $defaultLanguage Default language id
  * @param string $ids Multilingual div ids in form
  * @param string $id Current div id]
  * #param boolean $return define the return way : false for a display, true for a return
  */
function	displayFlags($languages, $defaultLanguage, $ids, $id, $return = false)
{
	if (sizeof($languages) == 1)
		return false;
	$defaultIso = Language::getIsoById($defaultLanguage);
	$output = '
	<div style="float: left; margin-left: 5px;">
		<img src="../img/flags/'.$defaultIso.'.png" class="pointer" id="language_current_'.$id.'" onclick="showLanguages(\''.$id.'\');" alt="" title="" />
	</div>
	<div id="languages_'.$id.'" class="language_flags">
		Choose language:<br /><br />';
	foreach ($languages as $language)
		$output .= '<img src="../img/flags/'.$language['iso_code'].'.png" class="pointer" alt="'.$language['name'].'" title="'.$language['name'].'" onclick="changeLanguage(\''.$id.'\', \''.$ids.'\', '.$language['id_lang'].', \''.$language['iso_code'].'\');" /> ';
	$output .= '</div>';
	
	if ($return)
		return $output;
	echo $output;
}

/**
  * Display SQL date in friendly format
  *
  * @param string $sqlDate Date in SQL format (YYYY-MM-DD HH:mm:ss)
  * @param boolean $withTime Display both date and time
  * @todo Several formats (french : DD-MM-YYYY)
  */
function	displayDate($sqlDate, $withTime = false)
{
	return strftime('%Y-%m-%d'.($withTime ? ' %H:%M:%S' : ''), strtotime($sqlDate));
}

/**
  * Check image upload
  *
  * @param array $file Upload $_FILE value
  * @param integer $maxFileSize Maximum upload size
  */
function	checkImage($file, $maxFileSize)
{
	if ($file['size'] > $maxFileSize)
		return Tools::displayError('image is too large').' ('.($file['size'] / 1000).'ko). '.Tools::displayError('Maximum allowed:').' '.($maxFileSize / 1000).'ko';
	/* pjpeg = Progressive jpeg */
	$types = array('image/gif', 'image/jpg', 'image/jpeg', 'image/pjpeg', 'image/png', 'image/x-png');
	if (!in_array($file['type'], $types))
		return Tools::displayError('image format not recognized, allowed formats are: .gif, .jpg, .png');
	if (strstr($file['name'], '.php') OR strstr($file['name'], '.pl'))
		return Tools::displayError('image format not recognized, allowed formats are: .gif, .jpg, .png');
	if ($file['error'])
		return Tools::displayError('error while uploading image; change your server\'s settings');
	return false;
}

/**
  * Check icon upload
  *
  * @param array $file Upload $_FILE value
  * @param integer $maxFileSize Maximum upload size
  */
function	checkIco($file, $maxFileSize)
{
	if ($file['size'] > $maxFileSize)
		return Tools::displayError('image is too large').' ('.($file['size'] / 1000).'ko). '.Tools::displayError('Maximum allowed:').' '.($maxFileSize / 1000).'ko';
	$types = array('image/x-icon');
	if (!in_array($file['type'], $types))
		return Tools::displayError('image format not recognized, allowed formats are: .ico');
	if (strstr($file['name'], '.php') OR strstr($file['name'], '.pl'))
		return Tools::displayError('image format not recognized, allowed formats are: .ico');
	if ($file['error'])
		return Tools::displayError('error while uploading image; change your server\'s settings');
	return false;
}

/**
  * Resize, cut and optimize image
  *
  * @param array $sourceFile Image object from $_FILE
  * @param string $destFile Destination filename
  * @param integer $destWidth Desired width (optional)
  * @param integer $destHeight Desired height (optional)
  *
  * @return boolean Operation result
  */
function	imageResize($sourceFile, $destFile, $destWidth = NULL, $destHeight = NULL, $fileType = 'jpg')
{
	if (!isset($sourceFile['tmp_name']) OR !file_exists($sourceFile['tmp_name']))
		return false;

	list($sourceWidth, $sourceHeight, $type, $attr) = @getimagesize($sourceFile['tmp_name']);
	if ($destWidth == NULL) $destWidth = $sourceWidth;
	if ($destHeight == NULL) $destHeight = $sourceHeight;

	switch ($type)
	{
		case 1:
			$sourceImage = @imagecreatefromgif($sourceFile['tmp_name']);
			break;
		case 2:
			$sourceImage = @imagecreatefromjpeg($sourceFile['tmp_name']);
			break;
		case 3:
			$sourceImage = @imagecreatefrompng($sourceFile['tmp_name']);
			break;
		default:
			return false;
	}
	
	$widthDiff = $destWidth - $sourceWidth;
	$heightDiff = $destHeight - $sourceHeight;
	
	if ($widthDiff > 0 AND $heightDiff > 0)
	{
		$nextWidth = $sourceWidth;
		$nextHeight = $sourceHeight;
	}
	else
	{
		if (intval(Configuration::get('PS_IMAGE_GENERATION_METHOD')) == 2 || (intval(Configuration::get('PS_IMAGE_GENERATION_METHOD')) == 0 && $widthDiff > $heightDiff))
		{
			$nextHeight = $destHeight;
			$nextWidth = intval(($sourceWidth * $nextHeight) / $sourceHeight);
			$destWidth = (intval(Configuration::get('PS_IMAGE_GENERATION_METHOD')) == 0 ? $destWidth : $nextWidth);
		}
		else
		{
			$nextWidth = $destWidth;
			$nextHeight = intval($sourceHeight * $destWidth / $sourceWidth);
			$destHeight = (intval(Configuration::get('PS_IMAGE_GENERATION_METHOD')) == 0 ? $destHeight : $nextHeight);
		}
	}

	$borderWidth = intval(($destWidth - $nextWidth) / 2);
	$borderHeight = intval(($destHeight - $nextHeight) / 2);
	
	$destImage = @imagecreatetruecolor($destWidth, $destHeight);

	$white = imagecolorallocate($destImage, 255, 255, 255);
	imagefill($destImage, 0, 0, $white);
	imagecopyresampled($destImage, $sourceImage, $borderWidth, $borderHeight, 0, 0, $nextWidth, $nextHeight, $sourceWidth, $sourceHeight);
	imagecolortransparent($destImage, $white);

	return ($fileType == 'gif' ? imagegif($destImage, $destFile) : ($fileType == 'png' ? imagepng($destImage, $destFile, 7) : imagejpeg($destImage, $destFile, 90)));
}

/**
  * Delete product or category image
  *
  * @param integer $id_item Product or category id
  * @param integer $id_image Image id
  */
function	deleteImage($id_item, $id_image = null)
{
	$path = ($id_image) ? _PS_PROD_IMG_DIR_ : _PS_CAT_IMG_DIR_;
	$table = ($id_image) ? 'product' : 'category';
	
	if (file_exists(_PS_TMP_IMG_DIR_.$table.'_'.$id_item.'.jpg'))
		unlink(_PS_TMP_IMG_DIR_.$table.'_'.$id_item.'.jpg');
	
	if ($id_image AND file_exists($path.$id_item.'-'.$id_image.'.jpg'))
		unlink($path.$id_item.'-'.$id_image.'.jpg');
	elseif (!$id_image AND file_exists($path.$id_item.'.jpg'))
		unlink($path.$id_item.'.jpg');
	
	$imagesTypes = ImageType::getImagesTypes();
	foreach ($imagesTypes AS $k => $imagesType)
		if ($id_image AND file_exists($path.$id_item.'-'.$id_image.'-'.$imagesType['name'].'.jpg'))
			unlink($path.$id_item.'-'.$id_image.'-'.$imagesType['name'].'.jpg');
		elseif (!$id_image AND file_exists($path.$id_item.'-'.$imagesType['name'].'.jpg'))
			unlink($path.$id_item.'-'.$imagesType['name'].'.jpg');
}

/**
  * Return path to a product category
  *
  * @param string $urlBase Start URL
  * @param integer $id_category Start category
  * @param string $path Current path
  * @param string $highlight String to highlight (in XHTML/CSS)
  */
function	getPath($urlBase, $id_category, $path = '', $highlight = '')
{
	global $cookie;
	
	$category = new Category($id_category, intval($cookie->id_lang));
	if (!$category->id)
		return $path;
	$name = ($highlight != NULL) ? str_ireplace($highlight, '<span class="highlight">'.$highlight.'</span>', 
	Category::hideCategoryPosition($category->name)) : Category::hideCategoryPosition($category->name);
	$edit = '<a href="'.$urlBase.'&id_category='.$category->id.'&addcategory"><img src="../img/admin/edit.gif" alt="Modify" /></a> ';
	if ($category->id == 1)
		$edit = '<a href="'.$urlBase.'&id_category='.$category->id.'&viewcategory"><img src="../img/admin/home.gif" alt="Home" /></a> ';
	$path = $edit.'<a href="'.$urlBase.'&id_category='.$category->id.'&viewcategory">'.$name.'</a> > '.$path;
	if ($category->id == 1)
		return substr($path, 0, strlen($path) - 3);
	return getPath($urlBase, $category->id_parent, $path);
}

?>