<?php
/**
  * Admin panel main class, AdminTab.php
  * Abstract class for admin panel tabs
  * @category admin
  *
  * @author PrestaShop <support@prestashop.com>
  * @copyright PrestaShop
  * @license http://www.opensource.org/licenses/osl-3.0.php Open-source licence 3.0
  * @version 0.8
  *
  */

abstract class AdminTab
{ 	
 	/** @var integer tab id */
 	public $id = -1;
 	
 	/** @var string tab name */
 	public $name;
	 
	/** @var boolean Automatically join language table if true */
 	public $lang = false;
 	
 	/** @var boolean Tab Automatically display edit/delete icons if true */
 	public $edit = false;
 	
 	/** @var boolean Tab Automatically display view icon if true */
 	public $view = false;
 	
 	/** @var boolean Table records are not deleted but marked as deleted */
 	public $deleted = false;
	
 	/** @var boolean Tab Automatically display duplicate icon if true */
 	public $duplicate = false;
 	
 	/** @var string Add fields into data query to display list */
 	protected $_select;
 	
 	/** @var string Join tables into data query to display list */
 	protected $_join;
 	
 	/** @var string Add conditions into data query to display list */
 	protected $_where;
 	
 	/** @var string Group rows into data query to display list */
 	protected $_group;
	
	/** @var array Name and directory where class image are located */
	public $fieldImageSettings = array();
	
	/** @var string Image type */
	public $imageType = 'jpg';
	
	/** @var array Fields to display in list */
	public $fieldsDisplay = array();
	
	/** @var array Cache for query results */
 	protected $_list = array();
 	
 	/** @var integer Number of results in list */
 	protected $_listTotal = 0;

	/** @var array WHERE clause determined by filter fields */
	protected $_filter = '';

	/** @var array Number of results in list per page (used in select field) */
	protected $_pagination = array(20, 50, 100, 300);

	/** @var string ORDER BY clause determined by field/arrows in list header */
 	protected $_orderBy;
 	
 	/** @var string Order way (ASC, DESC) determined by arrows in list header */
 	protected $_orderWay;

	/** @var integer Max image size for upload */
	protected $maxImageSize = 1000000;
	
	/** @var array Errors displayed after post processing */
 	protected $_errors = array();
	
	/** @var array Confirmations displayed after post processing */
	private $_conf;
	
	/** @var array tabAccess */
	public $tabAccess;
	
	/** @var bool Redirect or not ater a creation */
	protected $_redirect = true;
	
	private $_includeObj = array();
	
	public function __construct()
	{
		$this->id = Tab::getCurrentTabId();
		$this->_conf = array(
		1 => $this->l('Deletion successful'), 2 => $this->l('Selection successfully deleted'),
		3 => $this->l('Creation successful'), 4 => $this->l('Update successful'),
		5 => $this->l('Status update successful'), 6 => $this->l('Settings update successful'),
		7 => $this->l('Image successfully deleted'), 8 => $this->l('Module downloaded successfully'), 
		9 => $this->l('Thumbnails successfully regenerated'), 10 => $this->l('Message sent to the customer'),
		11 => $this->l('Comment added'), 12 => $this->l('Module installed successfully'), 
		13 => $this->l('Module uninstalled successfully'), 14 => $this->l('Language successfully copied'),
		15 => $this->l('Translations successfully added'), 16 => $this->l('Module grafted successfully to hook'),
		17 => $this->l('Module deleted successfully from hook')
		);
	}

	protected function l($string, $class = __CLASS__, $addslashes = FALSE)
	{
		global $_LANGADM;
		if (!is_array($_LANGADM))
			return str_replace('"', '&quot;', $string);
		$key = md5(addslashes($string));
		
		$str = (key_exists(get_class($this).$key, $_LANGADM)) ? $_LANGADM[get_class($this).$key] : ((key_exists($class.$key, $_LANGADM)) ? $_LANGADM[$class.$key] : $string);
		return str_replace('"', '&quot;', ($addslashes ? addslashes($str) : stripslashes($str)));
	}
	
	/**
	 * Manage page display (form, list...)
	 *
	 * @global string $currentIndex Current URL in order to keep current Tab
	 */
	public function display()
	{
		global $currentIndex, $cookie;
		
		// Include other tab in current tab
		if ($this->includeSubTab('display', array('submitAdd2', 'add', 'update', 'view')))
			;
		
		// Include current tab
		elseif ((Tools::getValue('submitAdd'.$this->table) AND sizeof($this->_errors)) OR isset($_GET['add'.$this->table]))
		{
			if ($this->tabAccess['add'] === '1')
			{
				$this->displayForm();
				echo '<br /><br /><a href="'.$currentIndex.'"><img src="../img/admin/arrow2.gif" /> '.$this->l('Back to list').'</a><br />';
			}
			else
				echo $this->l('You do not have permission to add here');
		}
		elseif (isset($_GET['update'.$this->table]))
		{
			if ($this->tabAccess['edit'] === '1')
			{
			 	$this->displayForm();
				echo '<br /><br /><a href="'.$currentIndex.'"><img src="../img/admin/arrow2.gif" /> '.$this->l('Back to list').'</a><br />';
			}
			else
				echo $this->l('You do not have permission to edit here');
		}
		elseif (isset($_GET['view'.$this->table]))
			$this->{'view'.$this->table}();

		else
		{
			$this->getList(intval($cookie->id_lang));
			$this->displayList();
			$this->displayOptionsList();
			$this->includeSubTab('display');
		}
	}
	
	public function includeSubTab($methodname, $actions = array())
	{
		if (!isset($this->_includeTab))
			return ;
		foreach ($this->_includeTab as $key => $subtab)
		{
			include_once('tabs/Admin'.$subtab.'.php');
			$classname = 'Admin'.$subtab;
			if (!isset($this->_includeObj[$key]))
				$this->_includeObj[$key] = new $classname;
			$AdminAddress = $this->_includeObj[$key];
			$inc = 0;
			foreach ($actions as $action)
			{
				switch ($action)
				{
					
					case 'submitAdd1':
						if (Tools::getValue('submitAdd'.$AdminAddress->table))
							$ok_inc = true;
						break;
					case 'submitAdd2':
						if (Tools::getValue('submitAdd'.$AdminAddress->table) AND sizeof($AdminAddress->_errors))
							$ok_inc = true;
						break;
					case 'submitDel':
						if (Tools::getValue('submitDel'.$AdminAddress->table))
							$ok_inc = true;
						break;
					case 'submitFilter':
						if (Tools::getValue('submitFilter'.$AdminAddress->table))
							$ok_inc = true;
					case 'submitReset':
						if (Tools::getValue('submitReset'.$AdminAddress->table))
							$ok_inc = true;
					default:
						if (isset($_GET[$action.$AdminAddress->table]))
							$ok_inc = true;
				}
			}
			if ((isset($ok_inc) AND $ok_inc) OR !count($actions))
			{
				if (!$AdminAddress->viewAccess())
				{
					echo Tools::displayError('access denied');
					return false;
				}
				if (!count($actions))
					if (($methodname == 'displayErrors' AND sizeof($AdminAddress->_errors)) OR $methodname != 'displayErrors')
						echo (isset($this->_includeTabTitle[$key]) ? '<h2>'.$this->_includeTabTitle[$key].'</h2>' : '');
				$AdminAddress->addressType = $this->table;
				$AdminAddress->$methodname();
				$inc++;
			}
		}
		return $inc;
	}
	
	/**
	 * Manage page display (form, list...)
	 *
	 * @param string $className Allow to validate a different class than the current one
	 */
	public function validateRules($className = false)
	{
		if (!$className)
			$className = $this->className;
			
		/* Class specific validation rules */
		$rules = call_user_func(array($className, 'getValidationRules'), $className);
	
		if ((sizeof($rules['requiredLang']) OR sizeof($rules['sizeLang']) OR sizeof($rules['validateLang']))
			AND method_exists($className, 'getTranslationsFieldsChild'))
		{
		 	/* Language() instance determined by default language */
			$defaultLanguage = new Language(intval(Configuration::get('PS_LANG_DEFAULT')));
			
			/* All availables languages */
			$languages = Language::getLanguages();
		}
		
		/* Checking for required fields */
		foreach ($rules['required'] AS $field)
			if (($value = Tools::getValue($field)) == false AND (string)$value != '0')
				if (!Tools::getValue('id_'.$this->table) OR ($field != 'passwd' AND $field != 'no-picture'))
					$this->_errors[] = $this->l('the field').' <b>'.call_user_func(array($className, 'displayFieldName'), $field, $className).'</b> '.$this->l('is required');	
		/* Checking for multilingual required fields */
		foreach ($rules['requiredLang'] AS $fieldLang)
			if (!Tools::getValue($fieldLang.'_'.$defaultLanguage->id))
				$this->_errors[] = $this->l('the field').' <b>'.call_user_func(array($className, 'displayFieldName'), $fieldLang, $className).'</b> '.$this->l('is required at least in').' '.$defaultLanguage->name;
	
		/* Checking for maximum fields sizes */
		foreach ($rules['size'] AS $field => $maxLength)
			if (Tools::getValue($field) AND strlen(Tools::getValue($field)) > $maxLength)
				$this->_errors[] = $this->l('the field').' <b>'.call_user_func(array($className, 'displayFieldName'), $field, $className).'</b> '.$this->l('is too long').' ('.$maxLength.' '.$this->l('chars max').')';

		/* Checking for maximum multilingual fields size */
		foreach ($rules['sizeLang'] AS $fieldLang => $maxLength)
			foreach ($languages AS $language)
				if (Tools::getValue($fieldLang.'_'.$language['id_lang']) AND strlen(Tools::getValue($fieldLang.'_'.$language['id_lang'])) > $maxLength)
					$this->_errors[] = $this->l('the field').' <b>'.call_user_func(array($className, 'displayFieldName'), $fieldLang, $className).' ('.$language['name'].')</b> '.$this->l('is too long').' ('.$maxLength.' '.$this->l('chars max').')';
		
		/* Overload this method for custom checking */
		$this->_childValidation();

		/* Checking for fields validity */
		foreach ($rules['validate'] AS $field => $function)
			if (Tools::getValue($field))
				if (!Validate::$function(Tools::getValue($field)))
					$this->_errors[] = $this->l('the field').' <b>'.call_user_func(array($className, 'displayFieldName'), $field, $className).'</b> '.$this->l('is invalid');
		
		/* Checking for multilingual fields validity */
		foreach ($rules['validateLang'] AS $fieldLang => $function)
			foreach ($languages AS $language)
				if (Tools::getValue($fieldLang.'_'.$language['id_lang']))
					if (!Validate::$function(Tools::getValue($fieldLang.'_'.$language['id_lang'])))
						$this->_errors[] = $this->l('the field').' <b>'.call_user_func(array($className, 'displayFieldName'), $fieldLang, $className).' ('.$language['name'].')</b> '.$this->l('is invalid');
	}
	
	/**
	 * Overload this method for custom checking
	 */
	protected function _childValidation() { }
	
	/**
	 * Overload this method for custom checking
	 *
	 * @param integer $id Object id used for deleting images
	 */
	private function deleteImage($id)
	{
	 	/* Deleting object images and thumbnails (cache) */
		if (!key_exists('dir', $this->fieldImageSettings))
			return false;
		$dir = $this->fieldImageSettings['dir'].'/';
		if (file_exists(_PS_IMG_DIR_.$dir.$id.'.'.$this->imageType))
			unlink(_PS_IMG_DIR_.$dir.$id.'.'.$this->imageType);
		if (file_exists(_PS_TMP_IMG_DIR_.$this->table.'_'.$id.'.'.$this->imageType))
			unlink(_PS_TMP_IMG_DIR_.$this->table.'_'.$id.'.'.$this->imageType);
		if (file_exists(_PS_TMP_IMG_DIR_.$this->table.'_mini_'.$id.'.'.$this->imageType))
			unlink(_PS_TMP_IMG_DIR_.$this->table.'_mini_'.$id.'.'.$this->imageType);
	}
	
	/**
	 * Manage page processing
	 *
	 * @global string $currentIndex Current URL in order to keep current Tab
	 */
	public function postProcess()
	{
		global $currentIndex;
		if (!isset($this->table))
			return false;

		// Sub included tab postProcessing
		if ($this->includeSubTab('postProcess', array('submitAdd1', 'submitDel', 'delete', 'submitFilter', 'submitReset')))
			;
			
		/* Delete object image */
		if (isset($_GET['deleteImage']))
		{
			if (Validate::isLoadedObject($object = $this->loadObject()) AND isset($this->fieldImageSettings))
			{
				$this->deleteImage($object->id);
				Tools::redirectAdmin($currentIndex.'&add'.$this->table.'&id_'.$this->table.'='.Tools::getValue('id_'.$this->table).'&conf=7');
			}
			else
				$this->_errors[] = Tools::displayError('an error occurred during image deletion (cannot load object)');
		}
		
		/* Delete object */
		elseif (isset($_GET['delete'.$this->table]))
		{
			if ($this->tabAccess['delete'] === '1')
		 	{
				if (Validate::isLoadedObject($object = $this->loadObject()) AND isset($this->fieldImageSettings))
				{
					// check if request at least one object with noZeroObject
					if (isset($object->noZeroObject) AND count(call_user_func(array($this->className, $object->noZeroObject))) <= 1)
						$this->_errors[] = Tools::displayError('you need at least one object').' <b>'.$this->table.'</b>'.Tools::displayError(', you cannot delete all of them');
					else
					{
						$this->deleteImage($object->id);
						if ($this->deleted)
						{
						 	$object->deleted = 1;
							if ($object->update())
								Tools::redirectAdmin($currentIndex.'&conf=1');
						}
						elseif ($object->delete())
							Tools::redirectAdmin($currentIndex.'&conf=1');
						$this->_errors[] = Tools::displayError('an error occurred during deletion');
					}
				}
				else
					$this->_errors[] = Tools::displayError('an error occurred while deleting object').' <b>'.$this->table.'</b> '.Tools::displayError('(cannot load object)');
			}
			else
				$this->_errors[] = Tools::displayError('You do not have permission to delete here.');
		}
		
		/* Change object statuts (active, inactive) */
		elseif (isset($_GET['status']))
		{
		 	if ($this->tabAccess['edit'] === '1')
			{
				if (Validate::isLoadedObject($object = $this->loadObject()))
				{
					if ($object->toggleStatus())
						Tools::redirectAdmin($currentIndex.'&conf=5'.((($id_category = intval(Tools::getValue('id_category'))) AND Tools::getValue('id_product')) ? '&id_category='.$id_category : ''));
					else
						$this->_errors[] = Tools::displayError('an error occurred while updating status');
				}
				else
					$this->_errors[] = Tools::displayError('an error occurred while updating status for object').' <b>'.$this->table.'</b> '.Tools::displayError('(cannot load object)');
			}
			else
				$this->_errors[] = Tools::displayError('You do not have permission to edit something here.');
		}
		
		/* Delete multiple objects */
		elseif(Tools::getValue('submitDel'.$this->table))
		{
		 	if ($this->tabAccess['delete'] === '1')
		 	{
			 	if (isset($_POST[$this->table.'Box']))
			 	{
					$object = new $this->className();
					if (isset($object->noZeroObject) AND
						// Check if all object will be deleted
						(count(call_user_func(array($this->className, $object->noZeroObject))) <= 1 OR count($_POST[$this->table.'Box']) == count(call_user_func(array($this->className, $object->noZeroObject)))))
						$this->_errors[] = Tools::displayError('you need at least one object').' <b>'.$this->table.'</b>'.Tools::displayError(', you cannot delete all of them');
					else
					{
						if ($object->deleteSelection($_POST[$this->table.'Box']))
							Tools::redirectAdmin($currentIndex.'&conf=2');
						$this->_errors[] = Tools::displayError('an error occurred while deleting selection');
					}
				}
				else
					$this->_errors[] = Tools::displayError('you must select at least one element to delete');
			}
			else
				$this->_errors[] = Tools::displayError('You do not have permission to delete here.');
		}
		
		/* Create or update an object */
		elseif(Tools::getValue('submitAdd'.$this->table))
		{
		 	/* Checking fields validity */
			$this->validateRules();
			if (!sizeof($this->_errors))
			{
				$id = intval(Tools::getValue('id_'.$this->table));

				/* Object update */
				if (isset($id) AND !empty($id))
				{
					if ($this->tabAccess['edit'] === '1')
					{
						$object = new $this->className($id);
						if (Validate::isLoadedObject($object))
						{
							/* Specific to objects which must not be deleted */
							if ($this->deleted AND $this->beforeDelete($object))
							{
								$object->deleted = 1;
								$object->update();
								$objectNew = new $this->className();
								$this->copyFromPost($objectNew, $this->table);
								$result = $objectNew->add();
								if (Validate::isLoadedObject($objectNew))
									$this->afterDelete($objectNew, $object->id);
							}
							else
							{
								$this->copyFromPost($object, $this->table);
								$result = $object->update();
							}
							if (!$result)
								$this->_errors[] = Tools::displayError('an error occurred while updating object').' <b>'.$this->table.'</b>';
							elseif ($this->postImage($object->id))
							{
								if ($back = Tools::getValue('back'))
									Tools::redirectAdmin(urldecode($back).'&conf=4');
								Tools::redirectAdmin($currentIndex.'&id_'.$this->table.'='.$object->id.'&conf=4');
							}
						}
						else
							$this->_errors[] = Tools::displayError('an error occurred while updating object').' <b>'.$this->table.'</b> '.Tools::displayError('(cannot load object)');
					}
					else
						$this->_errors[] = Tools::displayError('You do not have permission to edit something here.');
				}
				
				/* Object creation */
				else
				{
					if ($this->tabAccess['add'] === '1')
					{
						$object = new $this->className();
						$this->copyFromPost($object, $this->table);
						if (!$object->add())
							$this->_errors[] = Tools::displayError('an error occurred while creating object').' <b>'.$this->table.'</b>';
						elseif (($_POST['id_'.$this->table] = $object->id /* voluntary */) AND $this->postImage($object->id) AND $this->_redirect)
							Tools::redirectAdmin($currentIndex.'&id_'.$this->table.'='.$object->id.'&conf=3');
					}
					else
						$this->_errors[] = Tools::displayError('You do not have permission to add something here.');
				}
			}
		}
		
		/* Cancel all filters */
		elseif (isset($_POST['submitReset'.$this->table]))
			unset($_POST);
		
		/* Manage list filtering */
		elseif (Tools::isSubmit('submitFilter'.$this->table))
		{
		 	$object = new $this->className();
			
			foreach ($_POST AS $key => $value)
			{
			 	/* Extracting filters from $_POST on key filer_ */
				if ($value != NULL AND !strncmp($key, 'filter_', 7))
				{
				 	/* Table alias could be specified using a ! eg. alias!field */
					$key = substr($key, 7);
					$tmpTab = explode('!', $key);
					$key = isset($tmpTab[1]) ? $tmpTab[0].'.`'.$tmpTab[1].'`' : '`'.$tmpTab[0].'`';

					/* Only for date filtering (from, to) */
				 	if (is_array($value))
				 	{
				 	 	if (isset($value[0]) AND !empty($value[0]))
				 	 	{
				 	 	 	if (!Validate::isDate($value[0]))
				 	 			$this->_errors[] = Tools::displayError('\'from:\' date format is invalid (YYYY-MM-DD)');
				 	 		else
								$this->_filter .= ' AND '.pSQL($key).' >= \''.pSQL(Tools::dateFrom($value[0])).'\'';
						}
						
						if (isset($value[1]) AND !empty($value[1]))
				 	 	{
				 	 	 	if (!Validate::isDate($value[1]))
				 	 			$this->_errors[] = Tools::displayError('\'to:\' date format is invalid (YYYY-MM-DD)');
				 	 		else
								$this->_filter .= ' AND '.pSQL($key).' <= \''.pSQL(Tools::dateTo($value[1])).'\'';
						}
					}
					else
						$this->_filter .= ' AND '.(($key == 'id_'.$this->table OR $key == '`id_'.$this->table.'`') ? 'a.' : '').pSQL($key).' LIKE \'%'.pSQL($value).'%\' ';
				}
			}
		}

		/* Submit options list */
		elseif (Tools::getValue('submitOptions'.$this->table))
		{
			foreach ($this->_fieldsOptions as $key => $field)
				Configuration::updateValue($key, (isset($field['cast']) ? $field['cast'](Tools::getValue($key)) : Tools::getValue($key)));
			Tools::redirectAdmin($currentIndex.'&conf=6');
		}
	}
	
	protected function uploadImage($id, $name, $dir, $ext = false)
	{
	 	if (isset($_FILES[$name]['tmp_name']) AND !empty($_FILES[$name]['tmp_name']))
		{
		 	/* Delete old image */
			$this->deleteImage($id);

			/* Check image validity */
			if ($error = checkImage($_FILES[$name], $this->maxImageSize))
				$this->_errors[] = $error;
			
			/* Copy new image */
			else if (!imageResize($_FILES[$name], _PS_IMG_DIR_.$dir.$id.'.'.$this->imageType, NULL, NULL, ($ext ? $ext : $this->imageType)))
				$this->_errors[] = Tools::displayError('an error occurred while uploading image');
		}
		return !sizeof($this->_errors) ? true : false;
	}
	
	protected function uploadIco($name, $dest)
	{	
	
	 	if (isset($_FILES[$name]['tmp_name']) AND !empty($_FILES[$name]['tmp_name']))
		{
			/* Check ico validity */
			if ($error = checkIco($_FILES[$name], $this->maxImageSize))
				$this->_errors[] = $error;
			
			/* Copy new ico */
			elseif(!copy($_FILES[$name]['tmp_name'], $dest))
				$this->_errors[] = Tools::displayError('an error occurred while uploading favicon: '.$_FILES[$name]['tmp_name'].' to '.$dest);
		}
		return !sizeof($this->_errors) ? true : false;
	}
	
	/**
	 * Overload this method for custom checking
	 *
	 * @param integer $id Object id used for deleting images
	 * @return boolean
	 */
	protected function postImage($id)
	{
		if (isset($this->fieldImageSettings['name']) AND isset($this->fieldImageSettings['dir']))
			return $this->uploadImage($id, $this->fieldImageSettings['name'], $this->fieldImageSettings['dir'].'/');
		elseif (!empty($this->fieldImageSettings))
			foreach ($this->fieldImageSettings AS $image)
				if (isset($image['name']) AND isset($image['dir']))
					$this->uploadImage($id, $image['name'], $image['dir'].'/');
		return !sizeof($this->_errors) ? true : false;
	}
	
	/**
	 * Copy datas from $_POST to object
	 *
	 * @param object &$object Object
	 * @param string $table Object table
	 */
	protected function copyFromPost(&$object, $table)
	{
	 	/* Classical fields */
		foreach ($_POST AS $key => $value)
			if (key_exists($key, $object) AND $key != 'id_'.$table)
			{
			 	/* Do not take care of password field if empty */
			 	if ($key == 'passwd' AND Tools::getValue('id_'.$table) AND empty($value))
			 		continue;
			 	/* Automatically encrypt password in MD5 */
			 	if ($key == 'passwd' AND !empty($value))
			 		$value = Tools::encrypt($value);			 	
				$object->{$key} = $value;
			}
		
		/* Multilingual fields */
		$rules = call_user_func(array(get_class($object), 'getValidationRules'), get_class($object));
		if (sizeof($rules['validateLang']))
		{
			$languages = Language::getLanguages();
			foreach ($languages AS $language)
				foreach ($rules['validateLang'] AS $field => $validation)
					if (isset($_POST[$field.'_'.$language['id_lang']]))
						$object->{$field}[$language['id_lang']] = $_POST[$field.'_'.$language['id_lang']];
		}
	}
	
	/**
	 * Display errors
	 */
	public function displayErrors()
	{
		if ($this->includeSubTab('displayErrors'))
			;
		elseif ($nbErrors = sizeof($this->_errors))
		{
			echo '<div class="alert error"><h3>'.$nbErrors.' '.($nbErrors > 1 ? $this->l('errors') : $this->l('error')).'</h3>
			<ol>';
			foreach ($this->_errors AS $error)
				echo '<li>'.$error.'</li>';
			echo '
			</ol></div>';
		}
	}
	
	/**
	 * Display a warning message
	 *
	 * @param string $warn Warning message to display
	 */
	public function	displayWarning($warn)
	{
		echo '<div class="warning warn"><h3>'.$warn.'</h3></div>';
	}
	
	/**
	 * Display confirmations
	 */
	public function displayConf()
	{
		if ($conf = Tools::getValue('conf'))
			echo '<div class="conf confirm"><img src="../img/admin/ok.gif" /> '.$this->_conf[intval($conf)].'</div>';
	}
	
	/**
	 * Display a warning message
	 *
	 * @param integer $id_lang Language used for display
	 * @param string $orderBy ORDER BY clause
 	 * @param string $_orderWay Order way (ASC, DESC)
 	 * @param integer $start Offset in LIMIT clause
 	 * @param integer $limit Row count in LIMIT clause
	 */
	public function getList($id_lang, $orderBy = NULL, $orderWay = NULL, $start = 0, $limit = NULL)
	{
		global $cookie;
		
	 	/* Manage default params values */
	 	if (empty($limit))
			$limit = ((!isset($cookie->{$this->table.'_pagination'})) ? $this->_pagination[0] : $limit = $cookie->{$this->table.'_pagination'});
			
	 	if (!Validate::isTableOrIdentifier($this->table))
	 		die('filter is corrupted');
	 	if (empty($orderBy))
			$orderBy = Tools::getValue('orderby', 'id_'.$this->table);
	 	if (empty($orderWay))
			$orderWay = Tools::getValue('orderway', 'ASC');
		$limit = intval(Tools::getValue('pagination', $limit));
		$cookie->{$this->table.'_pagination'} = $limit;
		
		/* Check params validity */
		if (!Validate::isOrderBy($orderBy) OR !Validate::isOrderWay($orderWay) 
			OR !is_numeric($start) OR !is_numeric($limit)
			OR !Validate::isUnsignedId($id_lang))
			die(Tools::displayError('get list params is not valid'));		
		
		/* Determine offset from current page */
		if ((isset($_POST['submitFilter'.$this->table]) OR 
		isset($_POST['submitFilter'.$this->table.'_x']) OR 
		isset($_POST['submitFilter'.$this->table.'_y'])) AND 
		!empty($_POST['submitFilter'.$this->table]) AND 
		is_numeric($_POST['submitFilter'.$this->table]))
			$start = intval($_POST['submitFilter'.$this->table] - 1) * $limit;

		/* Cache */
		$this->_lang = intval($id_lang);
		$this->_orderBy = $orderBy;
		$this->_orderWay = strtoupper($orderWay);
		
		/* SQL table : orders, but class name is Order */
		$sqlTable = $this->table == 'order' ? 'orders' : $this->table;
		
		/* Query in order to get results number */
		$queryTotal = Db::getInstance()->getRow('
		SELECT COUNT(a.`id_'.$this->table.'`) AS total
		FROM `'._DB_PREFIX_.$sqlTable.'` a
		'.($this->lang ? 'LEFT JOIN `'._DB_PREFIX_.$this->table.'_lang` b ON (b.`id_'.$this->table.'` = a.`id_'.$this->table.'` AND b.`id_lang` = '.intval($id_lang).')' : '').' 
		'.(isset($this->_join) ? $this->_join.' ' : '').'
		WHERE 1 '.(isset($this->_where) ? $this->_where.' ' : '').(($this->deleted OR $this->table == 'currency') ? 'AND a.`deleted` = 0 ' : '').$this->_filter.' 
		'.(isset($this->_group) ? $this->_group.' ' : ''));
		$this->_listTotal = intval($queryTotal['total']);

		/* Query in order to get results with all fields */
		$this->_list = Db::getInstance()->ExecuteS('
		SELECT a.*'.($this->lang ? ', b.*' : '').(isset($this->_select) ? ', '.$this->_select.' ' : '').' 
		FROM `'._DB_PREFIX_.$sqlTable.'` a 
		'.($this->lang ? 'LEFT JOIN `'._DB_PREFIX_.$this->table.'_lang` b ON (b.`id_'.$this->table.'` = a.`id_'.$this->table.'` AND b.`id_lang` = '.intval($id_lang).')' : '').' 
		'.(isset($this->_join) ? $this->_join.' ' : '').'
		WHERE 1 '.(isset($this->_where) ? $this->_where.' ' : '').(($this->deleted OR $this->table == 'currency') ? 'AND a.`deleted` = 0 ' : '').$this->_filter.' 
		'.(isset($this->_group) ? $this->_group.' ' : '').'
		ORDER BY '.(($orderBy == 'id_'.$this->table) ? 'a.' : '').'`'.pSQL($orderBy).'` '.pSQL($orderWay).' 
		LIMIT '.intval($start).','.intval($limit));
	}
	
	/**
	 * Display image aside object form
	 *
	 * @param integer $id Object id
	 * @param string $image Local image filepath
 	 * @param integer $size Image width
 	 * @param integer $id_image Image id (for products with several images)
 	 *
 	 * @global string $currentIndex Current URL in order to keep current Tab
	 */
	public function displayImage($id, $image, $size, $id_image = null)
	{
		global $currentIndex;
		
		if ($id AND file_exists($image))
			echo '
			<div id="image">
				'.cacheImage($image, $this->table.'_'.intval($id).'.'.$this->imageType, $size, $this->imageType).'
				<p align="center">'.$this->l('Filesize').' '.(filesize($image) / 1000).'ko</p>
				<a href="'.$currentIndex.'&id_'.$this->table.'='.intval($id).'&deleteImage'.($id_image ? '&id_image='.intval($id_image) : '').'">
				<img src="../img/admin/delete.gif" alt="'.$this->l('Delete').'" /> '.$this->l('Delete').'</a>
			</div>';
	}
	
	/**
	 * Display list header (filtering, pagination and column names)
	 *
	 * @global string $currentIndex Current URL in order to keep current Tab
	 */
	public function displayListHeader()
	{
		global $currentIndex, $cookie;

		/* Determine total page number */
		$totalPages = ceil($this->_listTotal / Tools::getValue('pagination', (isset($cookie->{$this->table.'_pagination'}) ? $cookie->{$this->table.'_pagination'} : $this->_pagination[0])));
		if (!$totalPages) $totalPages = 1;
		echo '
		<form method="post" action="'.$currentIndex.'" class="form">
		<input type="hidden" id="submitFilter'.$this->table.'" name="submitFilter'.$this->table.'" value="0">
		<table>
			<tr>
				<td style="vertical-align: bottom;">
					<span style="float: left;">';
		
		/* Determine current page number */
		$page = intval(Tools::getValue('submitFilter'.$this->table));
		if (!$page) $page = 1;
		if ($page > 1)
			echo '
						<input type="image" src="../img/admin/list-prev2.gif" onclick="getE(\'submitFilter'.$this->table.'\').value=1" /> 
						&nbsp; <input type="image" src="../img/admin/list-prev.gif" onclick="getE(\'submitFilter'.$this->table.'\').value='.($page - 1).'" /> ';
		echo $this->l('Page').' <b>'.$page.'</b> / '.$totalPages;
		if ($page < $totalPages)
			echo '
						<input type="image" src="../img/admin/list-next.gif" onclick="getE(\'submitFilter'.$this->table.'\').value='.($page + 1).'" />
						 &nbsp;<input type="image" src="../img/admin/list-next2.gif" onclick="getE(\'submitFilter'.$this->table.'\').value='.$totalPages.'" /> ';
		echo '			| '.$this->l('Display').' 
						<select name="pagination">';
		
		/* Choose number of results per page */				
		foreach ($this->_pagination AS $value)
			echo '<option value="'.intval($value).'"'.((Tools::getValue('pagination', (isset($cookie->{$this->table.'_pagination'}) ? $cookie->{$this->table.'_pagination'} : null)) == $value) ? ' selected="selected"' : '').'>'.intval($value).'</option>';
		echo '
						</select>
					</span>
					<span style="float: right;">
						<input type="submit" name="submitReset'.$this->table.'" value="'.$this->l('Reset').'" class="button" />
						<input type="submit" id="submitFilter'.$this->table.'" name="submitFilter'.$this->table.'" value="'.$this->l('Filter').'" class="button" />
					</span>
					<span class="clear"></span>
				</td>
			</tr>
			<tr>
				<td>';

		/* Display column names and arrows for ordering (ASC, DESC) */
		echo '<table class="table" cellpadding="0" cellspacing="0"><tr>';
		if ($this->edit)
			echo '<th><input type="checkbox" name="checkme" class="noborder" onclick="checkDelBoxes(this.form, \''.$this->table.'Box[]\', this.checked)" /></th>';
		foreach ($this->fieldsDisplay AS $key => $params)
		{
			echo '
				<th>
					'.$params['title'];
			if (!isset($params['orderby']) OR $params['orderby'])
				echo '<br />
					<a href="'.$currentIndex.'&orderby='.urlencode($key).'&orderway=desc"><img border="0" src="../img/admin/down'.((isset($this->_orderBy) AND ($key == $this->_orderBy) AND ($this->_orderWay == 'DESC')) ? '_d' : '').'.gif" /></a>
					<a href="'.$currentIndex.'&orderby='.urlencode($key).'&orderway=asc"><img border="0" src="../img/admin/up'.((isset($this->_orderBy) AND ($key == $this->_orderBy) AND ($this->_orderWay == 'ASC')) ? '_d' : '').'.gif" /></a>';
			echo '
				</th>';
		}
		
		/* Check if object can be modified, deleted or detailed */
		if ($this->edit OR $this->view)
			echo '<th style="width: 52px">'.$this->l('Actions').'</th>';
		echo '</tr><tr style="height: 35px;">';
		
		if ($this->edit)
			echo '<td class="center">--</td>';
		
		/* Javascript hack in order to catch ENTER keypress event */
		$keyPress = 'onkeypress="formSubmit(event, \'submitFilter'.$this->table.'\');"';

		/* Filters (input, select, date or bool) */
		foreach ($this->fieldsDisplay AS $key => $params)
		{
		 	$width = (isset($params['width']) ? ' style="width: '.intval($params['width']).'px;"' : '');
		 	echo '<td'.(isset($params['align']) ? ' class="'.$params['align'].'"' : '').'>';
		 	if (!isset($params['type']))
		 		$params['type'] = 'text';
		 		
			$value = Tools::getValue('conf') ? null : Tools::getValue('filter_'.$key);
			if (isset($params['search']) AND !$params['search'])
			{
				echo '--</td>';
				continue;
			}
		 	switch ($params['type'])
		 	{
				case 'bool':
					echo '
					<select name="filter_'.$key.'">
						<option value="">--</option>
						<option value="1"'.($value == 1 ? ' selected="selected"' : '').'>'.$this->l('Yes').'</option>
						<option value="0"'.(($value == 0 AND $value != '') ? ' selected="selected"' : '').'>'.$this->l('No').'</option>
					</select>';
					break;
					
				case 'date':
					echo $this->l('From').' <input type="text" name="filter_'.(isset($params['filter_key']) ? $params['filter_key'] : $key).'[0]" value="'.(isset($value[0]) ? $value[0] : '').'"'.$width.' '.$keyPress.' /><br />
					'.$this->l('To').' <input type="text" name="filter_'.(isset($params['filter_key']) ? $params['filter_key'] : $key).'[1]" value="'.(isset($value[1]) ? $value[1] : '').'"'.$width.' '.$keyPress.' />';
					break;
					
				case 'select':
					
					if (isset($params['filter_key']))
					{
						echo '<select onchange="getE(\'submitFilter'.$this->table.'\').focus();getE(\'submitFilter'.$this->table.'\').click();" name="filter_'.$params['filter_key'].'" '.(isset($params['width']) ? 'style="width: '.$params['width'].'px"' : '').'>
								<option value=""'.(($value == 0 AND $value != '') ? ' selected="selected"' : '').'>--</option>';
						if (isset($params['select']) AND is_array($params['select']))
							foreach ($params['select'] AS $optionValue => $optionDisplay)
								echo '<option value="'.$optionValue.'"'.(((isset($_POST['filter_'.$params['filter_key']]) OR isset($_POST['filter_'.$params['filter_key']])) AND Tools::getValue('filter_'.$params['filter_key']) == $optionValue AND Tools::getValue('filter_'.$params['filter_key']) != '') ? ' selected="selected"' : '').'>'.$optionDisplay.'</option>';
						echo '</select>';
						break;
					}
					
				case 'text':
				default:
					echo '<input type="text" name="filter_'.(isset($params['filter_key']) ? $params['filter_key'] : $key).'" value="'.$value.'"'.$width.' '.$keyPress.' />';
			}			
			echo '</td>';
		}
		
		if ($this->edit OR $this->view)
			echo '<td class="center">--</td>';
		
		echo '</tr>';
	}
	
	/**
	 * Display list
 	 *
 	 * @global string $currentIndex Current URL in order to keep current Tab
	 */
	public function displayList()
	{		
		/* Do not copy category ID */
		$currentIndex2 = __PS_BASE_URI__.substr($_SERVER['PHP_SELF'], strlen(__PS_BASE_URI__)).'?tab='.Tools::getValue('tab');
		if ($this->edit)
			echo '<br /><a href="'.$currentIndex2.'&add'.$this->table.'"><img src="../img/admin/add.gif" border="0" /> '.$this->l('Add new').'</a><br /><br />';
		/* Append when we get a syntax error in SQL query */
		if ($this->_list === false)
		{
		 	$this->displayWarning($this->l('Bad SQL query'));
		 	return false;
		}
		
		/* Display list header (filtering, pagination and column names) */
		$this->displayListHeader();
		if (!sizeof($this->_list))
			echo '<tr><td class="center" colspan="'.sizeof($this->fieldsDisplay).'">'.$this->l('No items found').'</td></tr>';
		
		/* Show the content of the table */
		$this->displayListContent();
		
		/* Close list table and submit button */
		$this->displayListFooter();
	}
	
	public function displayListContent()
	{
		/* Display results in a table
		 *
		 * align  : determine value alignment
		 * prefix : displayed before value
		 * suffix : displayed after value
		 * image  : object image
		 * icon   : icon determined by values
		 * active : allow to toggle status
		 */
		global $currentIndex;
		$currency = new Currency(Configuration::get('PS_CURRENCY_DEFAULT'));

		$irow = 0;
		foreach ($this->_list AS $tr)
		{
			$id = $tr['id_'.$this->table];
		 	echo '<tr'.($irow++ % 2 ? ' class="alt_row"' : '').' '.(isset($tr['color']) ? 'style="background-color: '.$tr['color'].'"' : '').'>';
			if ($this->edit)
				echo '<td class="center"><input type="checkbox" name="'.$this->table.'Box[]" value="'.$id.'" class="noborder" /></td>';
			foreach ($this->fieldsDisplay AS $key => $params)
			{
			 	echo '
				<td style="cursor: pointer"'.(isset($params['align']) ? ' class="'.$params['align'].'"' : '').
					($this->view 
					? ' onclick="document.location = \''.$currentIndex.'&id_'.$this->table.'='.$id.'&view'.$this->table.'\'"' 
					: ' onclick="document.location = \''.$currentIndex.'&id_'.$this->table.'='.$id.'&update'.$this->table.'\'"').'>'.
					(isset($params['prefix']) ? $params['prefix'] : '');
				
				if (isset($params['active']) AND isset($tr[$key]))
					echo '<a href="'.$currentIndex.'&'.'id_'.$this->table.'='.$id.'&'.$params['active'].
					((($id_category = intval(Tools::getValue('id_category'))) AND Tools::getValue('id_product')) ? '&id_category='.$id_category : '').'">
					<img src="../img/admin/'.($tr[$key] ? 'enabled.gif' : 'disabled.gif').'" 
					alt="'.($tr[$key] ? $this->l('Enabled') : $this->l('Disabled')).'" title="'.($tr[$key] ? $this->l('Enabled') : $this->l('Disabled')).'" /></a>';
				elseif (isset($params['image']))
					echo cacheImage(_PS_IMG_DIR_.$params['image'].'/'.$id.'.'.$this->imageType, $this->table.'_mini_'.$id.'.'.$this->imageType, 45, $this->imageType);
				elseif (isset($params['icon']) AND isset($params['icon'][$tr[$key]]))
					echo '<img src="../img/admin/'.$params['icon'][$tr[$key]].'" alt="'.$tr[$key].'" title="'.$tr[$key].'" />';
				elseif (isset($params['price']))
					echo Tools::displayPrice($tr[$key], (isset($params['currency']) ? new Currency(intval($tr['id_currency'])) : $currency), false, false);
				elseif (isset($tr[$key]))
				{
					$echo = ($key == 'price' ? round($tr[$key], 2) : isset($params['maxlength']) ? substr($tr[$key], 0, $params['maxlength']).'...' : $tr[$key]);
					echo isset($params['callback']) ? call_user_func(array($this->className, $params['callback']), $echo) : $echo;
				}
				else
					echo '--';
				
				echo (isset($params['suffix']) ? $params['suffix'] : '').
				'</td>';
			}
			if ($this->edit OR $this->view)
			{
				echo '<td class="center">';
				if ($this->view)
					echo '
					<a href="'.$currentIndex.'&id_'.$this->table.'='.$id.'&view'.$this->table.'">
					<img src="../img/admin/details.gif" border="0" alt="'.$this->l('View').'" title="'.$this->l('View').'" /></a>';		
				if ($this->edit)
					echo '
					<a href="'.$currentIndex.'&id_'.$this->table.'='.$id.'&update'.$this->table.'">
					<img src="../img/admin/edit.gif" border="0" alt="'.$this->l('Edit').'" title="'.$this->l('Edit').'" /></a>
					<a href="'.$currentIndex.'&id_'.$this->table.'='.$id.'&delete'.$this->table.'" onclick="return confirm(\''.addslashes($this->l('Delete item #')).$id.' ?\');">
					<img src="../img/admin/delete.gif" border="0" alt="'.$this->l('Delete').'" title="'.$this->l('Delete').'" /></a>';
				$duplicate = $currentIndex.'&id_'.$this->table.'='.$id.'&duplicate'.$this->table;
				if ($this->duplicate)
					echo '
					<a style="cursor: pointer" onclick="if (confirm(\''.$this->l('Copy images too ?').'\')) document.location = \''.$duplicate.'\'; else document.location = \''.$duplicate.'&noimage=1\';">
					<img src="../img/admin/add.gif" border="0" alt="'.$this->l('Duplicate').'" title="'.$this->l('Duplicate').'" /></a>';
				echo '</td>';
			}
			echo '</tr>';
		}
	}
	
	/**
	 * Close list table and submit button
	 */
	public function displayListFooter()
	{
		echo '</table>';
		if ($this->edit)
			echo '<p><input type="submit" class="button" name="submitDel'.$this->table.'" value="'.$this->l('Delete selection').'" onclick="return confirm(\''.addslashes($this->l('Delete selected items?')).'\');" /></p>';
		echo '
				</td>
			</tr>
		</table>
		</form>';
		if (isset($this->_includeTab) AND count($this->_includeTab))
			echo '<br /><br />';
	}
	
	/**
	 * Options lists
	 */
	public function displayOptionsList()
	{
		global $currentIndex, $cookie, $tab;
		
		if (!isset($this->_fieldsOptions) OR !count($this->_fieldsOptions))
			return ;
		
		$tab = Tab::getTab(intval($cookie->id_lang), Tab::getIdFromClassName($tab));
		echo '<br /><br />';
		echo (isset($this->optionTitle) ? '<h2>'.$this->optionTitle.'</h2>' : '');
		echo '<form action="'.$currentIndex.'" id="'.$tab['name'].'" name="'.$tab['name'].'" method="post" class="width3">
			<fieldset>';
				echo (isset($this->optionTitle) ? '<legend><img src="../img/t/'.$tab['id_tab'].'.gif" />'.$this->optionTitle.'</legend>' : '');
		foreach ($this->_fieldsOptions AS $key => $field)
		{
			$val = Tools::getValue($key, Configuration::get($key));
			echo'
				<label>'.$field['title'].' </label>
				<div class="margin-form">';
					
			switch ($field['type'])
			{
				case 'select':
					echo '<select name="'.$key.'">';
					foreach ($field['list'] AS $value)
						echo '<option
							value="'.(isset($field['cast']) ? $field['cast']($value[$field['identifier']]) : $value[$field['identifier']]).'"'.($val == $value[$field['identifier']] ? ' selected="selected"' : '').'>'.$value['name'].'</option>';
					echo '</select>';
				break ;
				
				case 'bool':
					echo '<label class="t" for="'.$key.'_on"><img src="../img/admin/enabled.gif" alt="'.$this->l('Yes').'" title="'.$this->l('Yes').'" /></label>
					<input type="radio" name="'.$key.'" id="'.$key.'_on" value="1"'.($val ? ' checked="checked"' : '').' />
					<label class="t" for="'.$key.'_on"> '.$this->l('Yes').'</label>
					<label class="t" for="'.$key.'_off"><img src="../img/admin/disabled.gif" alt="'.$this->l('No').'" title="'.$this->l('No').'" style="margin-left: 10px;" /></label>
					<input type="radio" name="'.$key.'" id="'.$key.'_off" value="0" '.(!$val ? 'checked="checked"' : '').'/>
					<label class="t" for="'.$key.'_off"> '.$this->l('No').'</label>';
				break;
					
				default:
			}
			echo (isset($field['desc']) ? '<p>'.$field['desc'].'</p>' : '');
			echo '</div>';
		}
			echo '<div class="margin-form">
					<input type="submit" value="'.$this->l('   Save   ').'" name="submitOptions'.$this->table.'" class="button" />
				</div>
			</fieldset>
		</form>';
	}
	
	/**
	 * Load class object using identifier in $_GET (if possible)
	 * otherwise return an empty object, or die
 	 *
 	 * @param boolean $opt Return an empty object if load fail
 	 * @return object
	 */
	protected function loadObject($opt = false)
	{				
		if ($id = intval(Tools::getValue('id_'.$this->table)) AND Validate::isUnsignedId($id))
		{
			$object = new $this->className($id);
			if (!Validate::isLoadedObject($object))
				die (Tools::displayError('object cannot be loaded'));
			return $object;
		}
		else if ($opt)
			return new $this->className();
		else
			die(Tools::displayError('object cannot be loaded'));
	}
	
	/**
	 * Return field value if possible (both classical and multilingual fields)
	 *
	 * Case 1 : Return value if present in $_POST / $_GET
	 * Case 2 : Return object value
 	 *
 	 * @param object $obj Object
 	 * @param string $key Field name
 	 * @param integer $id_lang Language id (optional)
 	 * @return string
	 */
	protected function getFieldValue($obj, $key, $id_lang = null)
	{
		if ($id_lang)
			$defaultValue = ($obj->id AND isset($obj->{$key}[$id_lang])) ? $obj->{$key}[$id_lang] : '';
		else
			$defaultValue = isset($obj->{$key}) ? $obj->{$key} : '';
			
		return Tools::getValue($key.($id_lang ? '_'.$id_lang : ''), $defaultValue);
	}
	
	/**
	 * Display form
	 *
	 * @global string $currentIndex Current URL in order to keep current Tab
	 */
	public function displayForm() {	global $currentIndex; }
	
	/**
	 * Display object details
	 *
	 * @global string $currentIndex Current URL in order to keep current Tab
	 */
	public function viewDetails() {	global $currentIndex; }
		
	/**
	 * Called before deletion
 	 *
 	 * @param object $object Object
 	 * @return boolean
	 */
	protected function beforeDelete($object) { return true; }
	
	/**
	 * Called before deletion
 	 *
 	 * @param object $object Object
 	 * @return boolean
	 */
	protected function afterDelete($object, $oldId) { return true; }
	
	/**
	 * Check rights to view the current tab
 	 *
 	 * @return boolean
	 */
	public function viewAccess($disable = false)
	{
	 	global $cookie;

		if ($disable)
			return true;
	 	
	 	$access = new Access();
		$this->tabAccess = $access->getProfileAccess($cookie->profile, $this->id);
		
		if ($this->tabAccess['view'] === '1')
			return true;
		return false;
	}
}

?>
