<?php

/**
  * Cart class, Cart.php
  * Carts management
  * @category classes
  *
  * @author PrestaShop <support@prestashop.com>
  * @copyright PrestaShop
  * @license http://www.opensource.org/licenses/osl-3.0.php Open-source licence 3.0
  * @version 0.8
  *
  */

class		Cart extends ObjectModel
{
	public		$id;

	/** @var integer Customer delivery address ID */
	public 		$id_address_delivery;

	/** @var integer Customer invoicing address ID */
	public 		$id_address_invoice;

	/** @var integer Customer currency ID */
	public 		$id_currency;

	/** @var integer Customer ID */
	public 		$id_customer;

	/** @var integer Language ID */
	public 		$id_lang;

	/** @var integer Carrier ID */
	public 		$id_carrier;

	/** @var boolean True if the customer wants a recycled package */
	public		$recyclable = 1;

	/** @var boolean True if the customer wants a gift wrapping */
	public		$gift = 0;

	/** @var string Gift message if specified */
	public 		$gift_message;

	/** @var string Object creation date */
	public 		$date_add;

	/** @var string Object last modification date */
	public 		$date_upd;

	protected	$fieldsRequired = array('id_currency', 'id_lang');
	protected	$fieldsValidate = array('id_address_delivery' => 'isUnsignedId', 'id_address_invoice' => 'isUnsignedId',
		'id_currency' => 'isUnsignedId', 'id_customer' => 'isUnsignedId', 'id_lang' => 'isUnsignedId',
		'id_carrier' => 'isUnsignedId', 'recyclable' => 'isBool', 'gift' => 'isBool', 'gift_message' => 'isMessage');

	private		$_nb_products = NULL;
	private		$_products = NULL;
	private		$_discounts = NULL;
	protected 	$table = 'cart';
	protected 	$identifier = 'id_cart';

	public function getFields()
	{
		parent::validateFields();

		$fields['id_address_delivery'] = intval($this->id_address_delivery);
		$fields['id_address_invoice'] = intval($this->id_address_invoice);
		$fields['id_currency'] = intval($this->id_currency);
		$fields['id_customer'] = intval($this->id_customer);
		$fields['id_lang'] = intval($this->id_lang);
		$fields['id_carrier'] = intval($this->id_carrier);
		$fields['recyclable'] = intval($this->recyclable);
		$fields['gift'] = intval($this->gift);
		$fields['gift_message'] = pSQL($this->gift_message);
		$fields['date_add'] = pSQL($this->date_add);
		$fields['date_upd'] = pSQL($this->date_upd);

		return $fields;
	}

	public function add($autodate = true, $nullValues = false)
	{
		$return = parent::add($autodate);
		Module::hookExec('cart');
		return $return;
	}

	public function update($nullValues = false)
	{
		$return = parent::update();
		Module::hookExec('cart');
		return $return;
	}

	public function delete()
	{
		Db::getInstance()->Execute('DELETE FROM `'._DB_PREFIX_.'cart_discount` WHERE `id_cart` = '.intval($this->id));
		Db::getInstance()->Execute('DELETE FROM `'._DB_PREFIX_.'cart_product` WHERE `id_cart` = '.intval($this->id));
		return parent::delete();
	}

	/**
	 * Return cart discounts
	 *
	 * @result array Discounts
	 */
	public function getDiscounts($lite = false)
	{
		if (!$this->id)
			return array();
		if ($this->_discounts)
			return $this->_discounts;

		$result = Db::getInstance()->ExecuteS('
		SELECT d.*, `id_cart`
		FROM `'._DB_PREFIX_.'cart_discount` c
		LEFT JOIN `'._DB_PREFIX_.'discount` d ON c.`id_discount` = d.`id_discount`
		WHERE `id_cart` = '.intval($this->id));
		if ($lite)
			return $result;

		$total_products_wt = $this->getOrderTotal(true, 1);
		$shipping = $this->getOrderShippingCost();
		$this->_discounts = array();
		foreach ($result as $row)
		{
			$discount = new Discount($row['id_discount'], intval($this->id_lang));
			$row['description'] = $discount->description ? $discount->description : $discount->name;
			$row['value_real'] = number_format($discount->getValue(sizeof($result), $total_products_wt, $shipping), 2, '.', ' ');
			$this->_discounts[] = $row;
		}
		return $this->_discounts;
	}
	
	public function getDiscountsCustomer($id_discount)
	{
		$result = Db::getInstance()->ExecuteS('
		SELECT `id_discount`
		FROM `'._DB_PREFIX_.'cart_discount`
		WHERE `id_discount` = '.intval($id_discount).' AND `id_cart` = '.intval($this->id));

		return Db::getInstance()->NumRows();
	}

	/**
	 * Return cart products
	 *
	 * @result array Products
	 */
	public function getProducts()
	{
		if (!$this->id)
			return array();
		if ($this->_products)
			return $this->_products;
		$result = Db::getInstance()->ExecuteS('
		SELECT cp.`id_product_attribute`, cp.`id_product`, cp.`quantity` AS cart_quantity, pl.`name`, 
		pl.`description_short`, pl.`availability`, p.`id_product`, p.`id_supplier`, p.`id_manufacturer`, p.`id_tax`, p.`ean13`, p.`ecotax`,
		p.`quantity`, p.`price`, p.`reduction_price`, p.`reduction_percent`, p.`weight`, p.`out_of_stock`, p.`active`, p.`date_add`, p.`date_upd`,
		t.`id_tax`, tl.`name` AS tax, t.`rate`, pa.`price` AS price_attribute, pa.`weight` AS weight_attribute, pa.`quantity` AS quantity_attribute,
		i.`id_image`, il.`legend`, pl.`link_rewrite`, IF (IFNULL(pa.`reference`, \'\') = \'\', p.`reference`, pa.`reference`) AS reference
		FROM `'._DB_PREFIX_.'cart_product` cp
		LEFT JOIN `'._DB_PREFIX_.'product` p ON p.`id_product` = cp.`id_product`
		LEFT JOIN `'._DB_PREFIX_.'product_lang` pl ON (p.`id_product` = pl.`id_product` AND pl.`id_lang` = '.intval($this->id_lang).')
		LEFT JOIN `'._DB_PREFIX_.'product_attribute` pa ON (pa.`id_product_attribute` = cp.`id_product_attribute`)
		LEFT JOIN `'._DB_PREFIX_.'tax` t ON (t.`id_tax` = p.`id_tax`)
		LEFT JOIN `'._DB_PREFIX_.'tax_lang` tl ON (t.`id_tax` = tl.`id_tax` AND tl.`id_lang` = '.intval($this->id_lang).')
		LEFT JOIN `'._DB_PREFIX_.'image` i ON (i.`id_product` = cp.`id_product` AND (IF(pa.`id_image`, pa.`id_image` = i.`id_image`, i.`cover` = 1)))
		LEFT JOIN `'._DB_PREFIX_.'image_lang` il ON (i.`id_image` = il.`id_image` AND il.`id_lang` = '.intval($this->id_lang).')
		WHERE `id_cart` = '.intval($this->id).'
		AND p.`id_product` IS NOT NULL');

		/* Modify SQL results */
		$products = array();
		foreach ($result AS $k => $row)
		{
			$row['stock_quantity'] = intval($row['quantity']);
			$row['quantity'] = intval($row['cart_quantity']);
			$row['price'] = Product::getPriceStatic($row['id_product'], false, isset($row['id_product_attribute']) ? $row['id_product_attribute'] : NULL, 2);
			$row['price_wt'] = number_format(Product::getPriceStatic($row['id_product'], true, isset($row['id_product_attribute']) ? $row['id_product_attribute'] : NULL), 2, '.', '');
			$row['total'] = $row['price'] * $row['quantity'];
			$row['total_wt'] = number_format($row['price_wt'] * $row['quantity'], 2, '.', '');
			$row['id_image'] = Product::defineProductImage($row);
			$row['allow_oosp'] = Product::isAvailableWhenOutOfStock($row['out_of_stock']);
			$row['features'] = Product::getFeaturesStatic($row['id_product']);

			/* Add attributes to the SQL result if needed */
			if (isset($row['id_product_attribute']) AND Validate::isUnsignedInt($row['id_product_attribute']))
			{
				$result2 = Db::getInstance()->ExecuteS('
				SELECT agl.`public_name` AS public_group_name, al.`name` AS attribute_name
				FROM `'._DB_PREFIX_.'product_attribute_combination` pac
				LEFT JOIN `'._DB_PREFIX_.'attribute` a ON a.`id_attribute` = pac.`id_attribute`
				LEFT JOIN `'._DB_PREFIX_.'attribute_group` ag ON ag.`id_attribute_group` = a.`id_attribute_group`
				LEFT JOIN `'._DB_PREFIX_.'attribute_lang` al ON (a.`id_attribute` = al.`id_attribute` AND al.`id_lang` = '.intval($this->id_lang).')
				LEFT JOIN `'._DB_PREFIX_.'attribute_group_lang` agl ON (ag.`id_attribute_group` = agl.`id_attribute_group` AND agl.`id_lang` = '.intval($this->id_lang).')
				WHERE pac.`id_product_attribute` = '.$row['id_product_attribute']);

				$attributesList = '';
				$attributesListSmall = '';
				if ($result2)
					foreach ($result2 AS $k2 => $row2)
					{
						$attributesList .= $row2['public_group_name'].': '.$row2['attribute_name'].', ';
						$attributesListSmall .= $row2['attribute_name'].', ';
					}
				$attributesList = rtrim($attributesList, ', ');
				$attributesListSmall = rtrim($attributesListSmall, ', ');
				$row['attributes'] = $attributesList;
				$row['attributes_small'] = $attributesListSmall;
				$row['stock_quantity'] = $row['quantity_attribute'];
			}
			$products[] = $row;
		}
		$this->_products = $products;
		return $this->_products;
	}

	/**
	 * Return cart products quantity
	 *
	 * @result integer Products quantity
	 */
	public	function nbProducts()
	{
		if (!$this->id)
			return 0;
		if (!$this->_nb_products)
		{
			$row = Db::getInstance()->getRow('SELECT SUM(`quantity`) AS nb FROM `'._DB_PREFIX_.'cart_product` WHERE `id_cart` = '.intval($this->id));
			$this->_nb_products = intval($row['nb']);
		}
		return $this->_nb_products;
	}

	public static	function getNbProducts($id)
	{
		$row = Db::getInstance()->getRow('SELECT SUM(`quantity`) AS nb FROM `'._DB_PREFIX_.'cart_product` WHERE `id_cart` = '.intval($id));
		return intval($row['nb']);
	}

	/**
	 * Add a discount to the cart (NO controls except doubles)
	 *
	 * @param integer $id_discount The discount to add to the cart
	 * @result boolean Update result
	 */
	public	function addDiscount($id_discount)
	{
		return Db::getInstance()->AutoExecute(_DB_PREFIX_.'cart_discount', array('id_discount' => intval($id_discount), 'id_cart' => intval($this->id)), 'INSERT');
	}

	/**
	 * Update product quantity
	 *
	 * @param integer $quantity Quantity to add (or substract)
	 * @param integer $id_product Product ID
	 * @param integer $id_product_attribute Attribute ID if needed
	 * @param string $operator Indicate if quantity must be increased or decreased
	 */
	public	function updateQty($quantity, $id_product, $id_product_attribute = NULL, $operator = 'up')
	{
		if (intval($quantity) <= 0)
			return $this->deleteProduct(intval($id_product), intval($id_product_attribute));
		else
		{
			/* Check if the product is already in the cart */
			$result = Db::getInstance()->getRow('
			SELECT `quantity`
			FROM `'._DB_PREFIX_.'cart_product`
			WHERE `id_product` = '.intval($id_product).
			($id_product_attribute != NULL ? ' AND `id_product_attribute` = '.intval($id_product_attribute) : '').' AND `id_cart` = '.intval($this->id));

			/* Update quantity if product already exist */
			if (Db::getInstance()->NumRows())
			{
				if ($operator == 'up')
				{
					$result2 = Db::getInstance()->getRow('
						SELECT '.($id_product_attribute ? 'pa' : 'p').'.`quantity`, p.`out_of_stock`
						FROM `'._DB_PREFIX_.'product` p
						'.($id_product_attribute ? 'LEFT JOIN `'._DB_PREFIX_.'product_attribute` pa ON p.`id_product` = pa.`id_product`' : '').'
						WHERE p.`id_product` = '.intval($id_product).
						($id_product_attribute != NULL ? ' AND `id_product_attribute` = '.intval($id_product_attribute) : ''));
					$productQty = intval($result2['quantity']);
					$newQty = $result['quantity'] + intval($quantity);
					$qty = '`quantity` + '.intval($quantity);
					if ((intval($result2['out_of_stock']) == 0 OR ((intval($result2['out_of_stock']) == 2 AND !Configuration::get('PS_ORDER_OUT_OF_STOCK')))) AND $newQty > $productQty)
						return false;
				}
				elseif ($operator == 'down')
				{
					$qty = '`quantity` - '.intval($quantity);
					$newQty = $result['quantity'] - intval($quantity);
				}
				else
					return false;

				/* Delete product from cart */
				if ($newQty <= 0)
					return $this->deleteProduct(intval($id_product), intval($id_product_attribute));
				else
					Db::getInstance()->Execute('
					UPDATE `'._DB_PREFIX_.'cart_product`
					SET `quantity` = '.$qty.'
					WHERE `id_product` = '.intval($id_product).
					($id_product_attribute != NULL ? ' AND `id_product_attribute` = '.intval($id_product_attribute) : '').'
					AND `id_cart` = '.intval($this->id));
			}

			/* Add produt to the cart */
			else
			{
				$result2 = Db::getInstance()->getRow('
					SELECT '.($id_product_attribute ? 'pa' : 'p').'.`quantity`, p.`out_of_stock`
					FROM `'._DB_PREFIX_.'product` p
					'.($id_product_attribute ? 'LEFT JOIN `'._DB_PREFIX_.'product_attribute` pa ON p.`id_product` = pa.`id_product`' : '').'
					WHERE p.`id_product` = '.intval($id_product).
					($id_product_attribute != NULL ? ' AND `id_product_attribute` = '.intval($id_product_attribute) : ''));
					$productQty = intval($result2['quantity']);
				if (intval($quantity) > $productQty AND (intval($result2['out_of_stock']) == 0 OR (intval($result2['out_of_stock']) == 2 AND !Configuration::get('PS_ORDER_OUT_OF_STOCK'))))
					return false;
				return Db::getInstance()->AutoExecute(_DB_PREFIX_.'cart_product', array('id_product' => intval($id_product),
				'id_product_attribute' => intval($id_product_attribute), 'id_cart' => intval($this->id),
				'quantity' => intval($quantity)), 'INSERT');
			}
		}
		return true;
	}

	/**
	 * Check if order has already been placed
	 *
	 * @return boolean result
	 */
	public function OrderExists()
	{
		$result = Db::getInstance()->ExecuteS('SELECT `id_cart` FROM `'._DB_PREFIX_.'orders` WHERE `id_cart` = '.intval($this->id));
		return Db::getInstance()->NumRows();
	}

	/**
	 * Delete a discount from the cart
	 *
	 * @param integer $id_discount Discount ID
	 * @return boolean result
	 */
	public	function deleteDiscount($id_discount)
	{
		return Db::getInstance()->Execute('DELETE FROM `'._DB_PREFIX_.'cart_discount` WHERE `id_discount` = '.intval($id_discount).' AND `id_cart` = '.intval($this->id).' LIMIT 1');
	}

	/**
	 * Delete a product from the cart
	 *
	 * @param integer $id_product Product ID
	 * @param integer $id_product_attribute Attribute ID if needed
	 * @return boolean result
	 */
	public	function deleteProduct($id_product, $id_product_attribute = NULL)
	{
		return Db::getInstance()->Execute('DELETE FROM `'._DB_PREFIX_.'cart_product` WHERE `id_product` = '.intval($id_product).
		($id_product_attribute != NULL ? ' AND `id_product_attribute` = '.intval($id_product_attribute) : '').' AND `id_cart` = '.intval($this->id));
	}

	/**
	* This function returns the total cart amount
	*
	* type = 1 : only products
	* type = 2 : only discounts
	* type = 3 : both
	* type = 4 : both but without shipping
	* type = 5 : only shipping
	*
	* @param boolean $withTaxes With or without taxes
	* @param integer $type Total type
	* @return float Order total
	*/
	function getOrderTotal($withTaxes = true, $type = 3)
	{
		static $shipping_cost = -1;

		if (!$this->id)
			return 0;
		$type = intval($type);
		if (!in_array($type, array(1, 2, 3, 4, 5)))
			die(Tools::displayError());
			
		// no shipping cost if is a cart with only virtuals products
		if ($this->isVirtualCart() AND $type == 5)
			return 0;
		if ($this->isVirtualCart() AND $type == 3)
			$type = 4;
		$order_total = 0;
		if ($type != 4 AND $shipping_cost == -1)
			$shipping_cost = $this->getOrderShippingCost();
		$shipping_fees = $type != 4 ? $shipping_cost : 0;
		$products = $this->getProducts();
		foreach ($products AS $product)
		{
			$price = Product::getPriceStatic(intval($product['id_product']), $withTaxes, intval($product['id_product_attribute']));
			$total_price = $price * $product['quantity'];
			$order_total += $total_price;
		}
		$order_total_products = $order_total;

		if ($type == 2) $order_total = 0;
		if ($type != 1)
		{
			$discounts = $this->getDiscounts(true);
			foreach ($discounts AS $id_discount)
			{
				$discount = new Discount($id_discount['id_discount']);
				if ($discount->id)
				{
					$value = $discount->getValue(sizeof($discounts), $order_total_products, $shipping_fees);
					if ($discount->id_discount_type == 3)
					{
						if ($type == 2)
							$order_total -= $shipping_fees;
						$shipping_fees = 0;
					}
					else
						$order_total -= $value;
				}
			}
		}

		if ($type == 5) return $shipping_fees;
		if ($type == 3) $order_total += $shipping_fees;
		if ($order_total < 0 AND $type != 2) return 0;

		return $order_total;
	}

	/**
	* Return shipping total
	*
	* @param integer $id_carrier Carrier ID (default : current carrier)
	* @return float Shipping total
	*/
    function getOrderShippingCost($id_carrier = null)
    {
        $shipping_cost = 0;
        if (!$id_carrier)
            $id_carrier = $this->id_carrier;
        if (empty($id_carrier))
            return $shipping_cost;
        $carrier = new Carrier(intval($id_carrier));
        if (!$carrier->id OR !$carrier->active)
			return $shipping_cost;
		if ($carrier->id_tax AND Validate::isLoadedObject($tax = New Tax($carrier->id_tax)))
		{
			$address = new Address(intval($this->id_address_invoice));
			$country = new Country(intval($address->id_country));
			if (Tax::excludeTaxeByZone($tax->id, $country) OR Tax::excludeTaxeOption())
				$carrierTax = $tax->rate;
		}

		$configuration = Configuration::getMultiple(array('PS_SHIPPING_FREE_PRICE', 'PS_SHIPPING_HANDLING', 'PS_SHIPPING_METHOD', 'PS_SHIPPING_FREE_WEIGHT'));
        $orderTotal = $this->getOrderTotal(true, 4);
		if (isset($configuration['PS_SHIPPING_FREE_PRICE']) AND $orderTotal > floatval($configuration['PS_SHIPPING_FREE_PRICE']) AND floatval($configuration['PS_SHIPPING_FREE_PRICE']) > 0)
            return $shipping_cost;
		if (isset($configuration['PS_SHIPPING_FREE_WEIGHT']) AND $this->getTotalWeight() > floatval($configuration['PS_SHIPPING_FREE_WEIGHT']) AND floatval($configuration['PS_SHIPPING_FREE_WEIGHT']) > 0)
            return $shipping_cost;

        if (isset($this->id_address_delivery) AND $this->id_address_delivery)
			$id_zone = Address::getZoneById(intval($this->id_address_delivery));
		else
		{
			$defaultCountry = new Country(intval(Configuration::get('PS_COUNTRY_DEFAULT')));
			$id_zone = intval($defaultCountry->id_zone);
		}
        if (intval(Configuration::get('PS_SHIPPING_METHOD')))
            $shipping_cost += $carrier->getDeliveryPriceByWeight($this->getTotalWeight(), $id_zone);
        else
            $shipping_cost += $carrier->getDeliveryPriceByPrice($orderTotal, $id_zone);

		if (isset($carrierTax))
			 $shipping_cost /= 1 + ($carrierTax / 100);

		if (isset($configuration['PS_SHIPPING_HANDLING']) AND $carrier->shipping_handling)
            $shipping_cost += floatval($configuration['PS_SHIPPING_HANDLING']);

		return floatval($shipping_cost);
    }

	/**
	* Return cart weight
	*
	* @return float Cart weight
	*/
	function getTotalWeight()
	{
		if (!$this->id)
			return 0;

		$result = Db::getInstance()->getRow('
		SELECT SUM((p.`weight` + pa.`weight`) * cp.`quantity`) as nb
		FROM `'._DB_PREFIX_.'cart_product` cp
		LEFT JOIN `'._DB_PREFIX_.'product` p ON cp.`id_product` = p.`id_product`
		LEFT JOIN `'._DB_PREFIX_.'product_attribute` pa ON cp.`id_product_attribute` = pa.`id_product_attribute`
		WHERE (cp.`id_product_attribute` IS NOT NULL AND cp.`id_product_attribute` != 0)
		AND cp.`id_cart` = '.intval($this->id));
		$result2 = Db::getInstance()->getRow('
		SELECT SUM(p.`weight` * cp.`quantity`) as nb
		FROM `'._DB_PREFIX_.'cart_product` cp
		LEFT JOIN `'._DB_PREFIX_.'product` p ON cp.`id_product` = p.`id_product`
		WHERE (cp.`id_product_attribute` IS NULL OR cp.`id_product_attribute` = 0)
		AND cp.`id_cart` = '.intval($this->id));

		return floatval($result['nb'] + $result2['nb']);
	}

	/**
	* Check discount validity
	*
	* @return mixed Return a string if an error occurred and false otherwise
	*/
	function checkDiscountValidity($discountObj, $discounts, $order_total, $products, $checkCartDiscount = false)
	{
		global $cookie;

	    if (!$discountObj->active)
	        return Tools::displayError('this voucher has already been used or is disabled');
	    if (!$discountObj->quantity)
	        return Tools::displayError('this voucher has expired (usage limit attained)');
		if ($checkCartDiscount
			AND (
				$this->getDiscountsCustomer($discountObj->id) >= $discountObj->quantity_per_user
				OR	(Order::getDiscountsCustomer(intval($cookie->id_customer), $discountObj->id) + $this->getDiscountsCustomer($discountObj->id) >= $discountObj->quantity_per_user) >= $discountObj->quantity_per_user
				)
			)
			return Tools::displayError('you can\'t use more this voucher (usage limit attained)');
	    if (strtotime($discountObj->date_from) > time())
	        return Tools::displayError('this voucher is not yet valid');
	    if (strtotime($discountObj->date_to) < time())
	        return Tools::displayError('this voucher has expired');
	    if (!$discountObj->cumulable AND sizeof($discounts) > 1)
	        return Tools::displayError('this voucher isn\'t cumulative with other current discounts');
	    if (is_array($discounts) AND in_array($discountObj->id, $discounts))
	        return Tools::displayError('this voucher is already in your cart');
	    if ($discountObj->id_customer AND $this->id_customer != $discountObj->id_customer)
		{
			if (!$cookie->isLogged())
				Tools::redirect('authentication.php?back='.urlencode('order.php?submitDiscount&discount_name='.strval(Tools::getValue('discount_name'))));
			return Tools::displayError('you cannot use this voucher');
		}
	    if ($discountObj->minimal > floatval($order_total))
	        return Tools::displayError('the total amount of your order isn\'t high enough to use this voucher');
		if (!$discountObj->cumulable_reduction)
			foreach ($products as $product)
				if (intval($product['reduction_price']) OR intval($product['reduction_percent']))
					return Tools::displayError('this voucher isn\'t cumulative with products already bargained');
	    return false;
	}

	/**
	* Return useful informations for cart
	*
	* @return array Cart details
	*/
	function getSummaryDetails()
	{
		global $cookie;
		if (get_class($cookie) != 'Cookie')
			die (Tools::displayError());

		return array(
			'delivery' => new Address(intval($this->id_address_delivery)),
			'invoice' => new Address(intval($this->id_address_invoice)),
			'carrier' => new Carrier(intval($this->id_carrier), $cookie->id_lang),
			'products' => $this->getProducts(intval($cookie->id_lang)),
		  'discounts' => $this->getDiscounts(),
			'total_discounts' => number_format($this->getOrderTotal(true, 2), 2, '.', ''),
			'total_shipping' => number_format($this->getOrderTotal(true, 5), 2, '.', ''),
			'total_products' => number_format($this->getOrderTotal(false, 1), 2, '.', ''),
			'total_products_wt' => number_format($this->getOrderTotal(true, 1), 2, '.', ''),
			'total_price' => number_format($this->getOrderTotal(), 2, '.', ''));
	}

	/**
	* Return carts thats have not been converted in orders
	*
	* @param string $dateFrom Select only cart updated after this date
	* @param string $dateTo Select only cart updated before this date
	* @return array Carts
	*/
	static function getNonOrderedCarts($dateFrom, $dateTo)
	{
		if (!Validate::isDate($dateFrom) OR !Validate::isDate($dateTo))
			die (Tools::displayError());

		return Db::getInstance()->ExecuteS('
		SELECT cart.`id_cart`, cart.`date_upd`, c.`id_customer` AS id_customer, c.`name` AS customer_name, c.`surname` AS customer_surname,
		SUM(cp.`quantity`) AS nb_products,
		COUNT(cd.`id_cart`) AS nb_discounts
		FROM `'._DB_PREFIX_.'cart` cart
		LEFT JOIN `'._DB_PREFIX_.'cart_product` cp ON cart.`id_cart` = cp.`id_cart`
		LEFT JOIN `'._DB_PREFIX_.'cart_discount` cd ON cart.`id_cart` = cd.`id_cart`
		LEFT JOIN `'._DB_PREFIX_.'customer` c ON cart.`id_customer` = c.`id_customer`
		WHERE cart.`id_cart` NOT IN (SELECT `id_cart` FROM `'._DB_PREFIX_.'orders`)
		AND TO_DAYS(cart.`date_upd`) >= TO_DAYS(\''.pSQL(strftime('%Y-%m-%d %H:%M:%S', strtotime($dateFrom))).'\')
		AND TO_DAYS(cart.`date_upd`) <= TO_DAYS(\''.pSQL(strftime('%Y-%m-%d %H:%M:%S', strtotime($dateTo))).'\')
		GROUP BY cart.`id_cart`, cp.`id_cart`, cd.`id_cart`
		ORDER BY cart.`date_upd` DESC');
	}

	/**
	* Return customer e-mail
	*
	* @return string Customer e-mail
	*/
	function getCustomeremail()
	{
		if (!$this->id)	return false;

		$result = Db::getInstance()->getRow('
		SELECT cu.`email`
		FROM `'._DB_PREFIX_.'cart` ca
		LEFT JOIN `'._DB_PREFIX_.'customer` cu ON ca.`id_customer` = cu.`id_customer`
		WHERE ca.`id_cart` = '.intval($this->id));

		if (!Validate::isemail($result['email']))
			die(Tools::displayError());

		return $result['email'];
	}

	public function checkQuantities()
	{
		if (Configuration::get('PS_STOCK_MANAGEMENT'))
			foreach ($this->getProducts() AS $product)
			    if (!$product['allow_oosp'] AND $product['stock_quantity'] < $product['quantity'])
			    	return false;
		return true;
	}

	static public function lastNoneOrderedCart($id_customer)
	{
	 	if (!$result = Db::getInstance()->getRow('
		 	SELECT c.`id_cart`
			FROM '._DB_PREFIX_.'cart c
			LEFT JOIN '._DB_PREFIX_.'orders o ON (c.`id_cart` = o.`id_cart`)
			WHERE c.`id_customer` = '.intval($id_customer).' AND o.`id_cart` IS NULL
			ORDER BY c.`date_upd` DESC'))
	 		return false;
	 	return $result['id_cart'];
	}

	/**
	* Check if cart contains only virtual products
	* @return boolean true if is a virtual cart or false
	*
	*/
	public function isVirtualCart()
	{
		if (!$this->_nb_products) return false;
		$allVirtual = true;
		foreach ($this->getProducts() AS $product)
		{
			$allVirtual &= (Validate::isUnsignedInt(ProductDownload::getIdFromIdProduct($product['id_product'])) ? true : false);
		}
		return $allVirtual;
	}

	static public function getCartByOrderId($id_order)
	{
		$result = Db::getInstance()->getRow('SELECT `id_cart` FROM '._DB_PREFIX_.'orders WHERE `id_order` = '.intval($id_order));
		if (!$result OR empty($result) OR !key_exists('id_cart', $result))
			return false;
		return new Cart(intval($result['id_cart']));
	}

}
