<?php

/**
  * Discount class, Discount.php
  * Discounts management
  * @category classes
  *
  * @author PrestaShop <support@prestashop.com>
  * @copyright PrestaShop
  * @license http://www.opensource.org/licenses/osl-3.0.php Open-source licence 3.0
  * @version 0.8
  *
  */
  
class		Discount extends ObjectModel
{
	public		$id;
	
	/** @var integer Customer id only if discount is reserved */
	public		$id_customer;
	
	/** @var integer Discount type ID */
	public		$id_discount_type;
	
	/** @var string Name (the one which must be entered) */
	public 		$name;
	
	/** @var string A short description for the discount */
	public 		$description;
	
	/** @var string Value in percent as well as in euros */
	public 		$value;
	
	/** @var integer Totale quantity available */
	public 		$quantity;
	
	/** @var integer User quantity available */
	public 		$quantity_per_user;
	
	/** @var boolean Indicate if discount is cumulable with others */
	public 		$cumulable;
	
	/** @var integer Indicate if discount is cumulable with already bargained products */
	public 		$cumulable_reduction;
	
	/** @var integer Date from wich discount become active */
	public 		$date_from;
	
	/** @var integer Date from wich discount is no more active */
	public 		$date_to;
	
	/** @var integer Minimum cart total amount required to use the discount */
	public 		$minimal;
	
	/** @var boolean Status */
	public 		$active = true;
	
	protected	$fieldsRequired = array('id_discount_type', 'name', 'value', 'quantity', 'quantity_per_user', 'date_from', 'date_to');
	protected	$fieldsSize = array('name' => '32', 'date_from' => '32', 'date_to' => '32');
	protected	$fieldsValidate = array('id_customer' => 'isUnsignedId', 'id_discount_type' => 'isUnsignedId',
		'name' => 'isDiscountName', 'value' => 'isPrice', 'quantity' => 'isUnsignedInt', 'quantity_per_user' => 'isUnsignedInt',
		'cumulable' => 'isBool', 'cumulable_reduction' => 'isBool', 'date_from' => 'isDate',
		'date_to' => 'isDate', 'minimal' => 'isFloat', 'active' => 'isBool');
	protected	$fieldsRequiredLang = array('description');
	protected	$fieldsSizeLang = array('description' => 128);
	protected	$fieldsValidateLang = array('description' => 'isGenericName');

	protected 	$table = 'discount';
	protected 	$identifier = 'id_discount';
		
	public function getFields()
	{
		parent::validateFields();
		
		$fields['id_customer'] = intval($this->id_customer);
		$fields['id_discount_type'] = intval($this->id_discount_type);
		$fields['name'] = pSQL($this->name);
		$fields['value'] = floatval($this->value);
		$fields['quantity'] = intval($this->quantity);
		$fields['quantity_per_user'] = intval($this->quantity_per_user);
		$fields['cumulable'] = intval($this->cumulable);
		$fields['cumulable_reduction'] = intval($this->cumulable_reduction);
		$fields['date_from'] = pSQL($this->date_from);
		$fields['date_to'] = pSQL($this->date_to);
		$fields['minimal'] = intval($this->minimal);
		$fields['active'] = intval($this->active);

		return $fields;
	}
	
	public function getTranslationsFieldsChild()
	{
		parent::validateFieldsLang();
		return parent::getTranslationsFields(array('description'));
	}
	
	/**
	  * Return discount types list
	  *
	  * @return array Discount types
	  */
	static public function getDiscountTypes($id_lang)
	{
		return Db::getInstance()->ExecuteS('
		SELECT *
		FROM '._DB_PREFIX_.'discount_type dt
		LEFT JOIN `'._DB_PREFIX_.'discount_type_lang` dtl ON (dt.`id_discount_type` = dtl.`id_discount_type` AND dtl.`id_lang` = '.intval($id_lang).')');
	}
    
	/**
	  * Get discount ID from name
	  *
	  * @param string $discountName Discount name
	  * @return integer Discount ID
	  */
	public static function getIdByName($discountName)
	{
	 	if (!Validate::isDiscountName($discountName))
	 		die(Tools::displayError());
	 		
		$result = Db::getInstance()->getRow('
		SELECT `id_discount`
		FROM `'._DB_PREFIX_.'discount`
		WHERE `name` = \''.pSQL($discountName).'\'');
		return isset($result['id_discount']) ? $result['id_discount'] : false;
	}
	
	/**
	  * Return customer discounts
	  *
	  * @param integer $id_lang Language ID
	  * @param boolean $id_customer Customer ID
	  * @return array Discounts
	  */
	static public function getCustomerDiscounts($id_lang, $id_customer)
    {
    	return Db::getInstance()->ExecuteS('
        SELECT d.*, dtl.`name` AS `type`
		FROM `'._DB_PREFIX_.'discount` d
		LEFT JOIN `'._DB_PREFIX_.'discount_lang` dl ON (d.`id_discount` = dl.`id_discount` AND dl.`id_lang` = '.intval($id_lang).')
		LEFT JOIN `'._DB_PREFIX_.'discount_type` dt ON dt.`id_discount_type` = d.`id_discount_type`
		LEFT JOIN `'._DB_PREFIX_.'discount_type_lang` dtl ON (dt.`id_discount_type` = dtl.`id_discount_type` AND dtl.`id_lang` = '.intval($id_lang).')
		WHERE `id_customer` = '.intval($id_customer));
    }
	
	/**
	  * Return discount value
	  *
	  * @param integer $nb_discounts Number of discount currently in cart
	  * @param boolean $order_total_products Total cart products amount
	  * @return mixed Return a float value or '!' if reduction is 'Shipping free'
	  */
	function getValue($nb_discounts = 0, $order_total_products = 0)
    {
        if ($this->minimal > 0 AND intval($order_total_products) < $this->minimal) return 0;
        if (!$this->cumulable AND intval($nb_discounts) > 1) return 0;
		if (!$this->active) return 0;
		if (!$this->quantity) return 0;
		
        $date_start = strtotime($this->date_from);
        $date_end = strtotime($this->date_to);
        if (time() < $date_start OR time() > $date_end) return 0;

        switch ($this->id_discount_type)
        {
            case 1:
                // %
                $percentage = $this->value / 100;
                return $percentage * intval($order_total_products);
            case 2:
                // amount
                return $this->value;
            case 3:
                // Shipping is free
                return '!';
        }
		return 0;
    }

	static public function discountExists($discountName, $id_discount = 0)
	{
		return Db::getInstance()->getRow('SELECT `id_discount` FROM '._DB_PREFIX_.'discount WHERE `name` LIKE \''.$discountName.'\' AND `id_discount` != '.intval($id_discount));
	}
}

?>