<?php

/**
  * Module class, Module.php
  * Modules management
  * @category classes
  *
  * @author PrestaShop <support@prestashop.com>
  * @copyright PrestaShop
  * @license http://www.opensource.org/licenses/osl-3.0.php Open-source licence 3.0
  * @version 0.9
  *
  */

abstract class Module
{
	/** @var integer Module ID */
	public $id = NULL;

	/** @var float Version */
	public $version;

	/** @var string Unique name */
	public $name;

	/** @var string Human name */
	public $displayName;

	/** @var string A little description of the module */
	public $description;

	/** @var string Admin tab correponding to the module */
	public $tab = NULL;

	/** @var boolean Status */
	public $active = false;

	/** @var array current language translations */
	private $_lang = array();

	/** @var string Module web path (eg. '/shop/modules/modulename/')  */
	protected $_path = NULL;
	
	/** @var string Fill it if the module is installed but not yet set up */
	public $warning;

	protected $table = 'module';

	protected $identifier = 'id_module';

	protected $page;

	static public $_db;
	
	/*
	  * Constructor
	  * 
	  * @param string $name Module unique name
	  */
	public function __construct($name = NULL)
	{
		global $cookie;

		if ($this->name == NULL)
			$this->name = $this->id;
		if ($this->name != NULL)
		{
			$result = Db::getInstance()->GetRow('SELECT * FROM `'.pSQL(_DB_PREFIX_.$this->table).'` WHERE `name` = \''.pSQL($this->name).'\'');
			if (!$result)
				return false;
			$this->active = true;
			$this->id = $result['id_module'];
			foreach ($result AS $key => $value)
				if (key_exists($key, $this))
					$this->{$key} = $value;
			$this->_path = __PS_BASE_URI__.'modules/'.$this->name.'/';
		}
	}

	/*
	  * Insert module into datable
	  */
	public function install()
	{
		if (!Validate::isModuleName($this->name))
			die(Tools::displayError());

		$result = Db::getInstance()->getRow('
				SELECT `id_module`
				FROM `'._DB_PREFIX_.'module`
				WHERE `name` = \''.pSQL($this->name).'\'');
		if ($result)
			return false;
		$result = Db::getInstance()->AutoExecute(_DB_PREFIX_.$this->table, array('name' => $this->name, 'active' => 1), 'INSERT');
		if (!$result)
			return false;
		$this->id = Db::getInstance()->Insert_ID();
		return true;
	}

	/*
	 * Delete module from datable
	 *
	 * @return boolean result
	 */
	public function uninstall()
	{
		if (!Validate::isUnsignedId($this->id))
			return false;
		$result = Db::getInstance()->ExecuteS('
				SELECT `id_hook`
				FROM `'._DB_PREFIX_.'hook_module` hm
				WHERE `id_module` = '.$this->id);
		foreach	($result AS $row)
		{
			Db::getInstance()->Execute('
					DELETE FROM `'._DB_PREFIX_.'hook_module`
					WHERE `id_module` = '.intval($this->id).'
					AND `id_hook` = '.intval($row['id_hook']));
			$this->cleanPositions($row['id_hook']);
		}
		return Db::getInstance()->Execute('
				DELETE FROM `'._DB_PREFIX_.'module` 
				WHERE `id_module` = '.intval($this->id));
	}

	/*
	  * Connect module to a hook
	  *
	  * @param string $hook_name Hook name
	  * @return boolean result
	  */
	public function registerHook($hook_name)
	{
	 	if (!Validate::isHookName($hook_name))
	 		die(Tools::displayError());
	 	if (!isset($this->id) OR !is_numeric($this->id))
			return false;

		// Check if already register
		$result = Db::getInstance()->getRow('
		SELECT hm.`id_module` FROM `'._DB_PREFIX_.'hook_module` hm, `'._DB_PREFIX_.'hook` h
		WHERE hm.`id_module` = '.intval($this->id).'
		AND h.`name` = \''.pSQL($hook_name).'\'
		AND h.`id_hook` = hm.`id_hook`');
		if ($result)
			return true;
		
		// Get hook id
		$result = Db::getInstance()->getRow('
		SELECT `id_hook`
		FROM `'._DB_PREFIX_.'hook`
		WHERE `name` = \''.pSQL($hook_name).'\'');
		if (!isset($result['id_hook']))
			return false;
		
		// Get module position in hook
		$result2 = Db::getInstance()->getRow('
		SELECT MAX(`position`) AS position
		FROM `'._DB_PREFIX_.'hook_module`
		WHERE `id_hook` = '.intval($result['id_hook']));
		if (!$result2)
			return false;

		// Register module in hook
		$return = Db::getInstance()->Execute('
		INSERT INTO `'._DB_PREFIX_.'hook_module` (`id_module`, `id_hook`, `position`)
		VALUES ('.intval($this->id).', '.intval($result['id_hook']).', '.intval($result2['position'] + 1).')');
		
		$this->cleanPositions(intval($result['id_hook']));
		
		return $return;
	}
	
	
	/*
	  * Unregister module from hook
	  *
	  * @param int $id_hook Hook id
	  * @return boolean result
	  */
	public function unregisterHook($hook_id)
	{
		return Db::getInstance()->Execute('
		DELETE
		FROM `'._DB_PREFIX_.'hook_module`
		WHERE `id_module` = '.intval($this->id).'
		AND `id_hook` = '.intval($hook_id));
	}
	
	/*
	  * Unregister exceptions linked to module
	  *
	  * @param int $id_hook Hook id
	  * @return boolean result
	  */
	public function unregisterExceptions($hook_id)
	{
		return Db::getInstance()->Execute('
		DELETE
		FROM `'._DB_PREFIX_.'hook_module_exceptions`
		WHERE `id_module` = '.intval($this->id).'
		AND `id_hook` = '.intval($hook_id));
	}

	/*
	  * Add exceptions for module->Hook
	  *
	  * @param int $id_hook Hook id
	  * @param array $excepts List of file name
	  * @return boolean result
	  */
	public function registerExceptions($id_hook, $excepts)
	{
		foreach ($excepts AS $except) {
			if (!empty($except))
			{
				$result = Db::getInstance()->Execute('
				INSERT INTO `'._DB_PREFIX_.'hook_module_exceptions` (`id_module`, `id_hook`, `file_name`)
				VALUES ('.intval($this->id).', '.intval($id_hook).', \''.strval($except).'\')');
				if (!$result)
					return false;
			}
		}
		return true;
	}
	
	/*
	  * Return an instance of the specified module
	  *
	  * @param string $moduleName Module name
	  * @return Module instance
	  */
	static public function getInstanceByName($moduleName)
    {
     	if (!Validate::isModuleName($moduleName))
     		die(Tools::displayError());
		if (!file_exists(_PS_MODULE_DIR_.$moduleName.'/'.$moduleName.'.php'))
			return false;
        include_once(_PS_MODULE_DIR_.$moduleName.'/'.$moduleName.'.php');
        if (!class_exists($moduleName, false))
			return false;
        return (new $moduleName);
    }
    
	/*
	  * Return an instance of the specified module
	  *
	  * @param integer $id_module Module ID
	  * @return Module instance
	  */
    static public function getInstanceById($id_module)
    {
    	$result = Db::getInstance()->GetRow('
		SELECT `name`
		FROM `'._DB_PREFIX_.'module`
		WHERE `id_module` = '.intval($id_module));

		return ($result ? Module::getInstanceByName($result['name']) : false);
    }
    
	/*
	  * Return available modules
	  *
	  * @return array Modules
	  */
    public static function getModulesOnDisk()
    {
		$modules_dir = self::getModulesDirOnDisk();
		foreach ($modules_dir AS $module)
		{
			require_once _PS_MODULE_DIR_.'/'.$module.'/'.$module.'.php';
			$moduleList[] = new $module;
		}
		return $moduleList;
    }
	
	public static function getModulesDirOnDisk()
	{
		$moduleList = array();
        $modules = scandir(_PS_MODULE_DIR_);
        foreach ($modules AS $name)
        {
	        if (file_exists($moduleFile = _PS_MODULE_DIR_.'/'.$name.'/'.$name.'.php'))
	        {
	         	if (!Validate::isModuleName($name))
	         		die(Tools::displayError().' (Module '.$name.')');
				$moduleList[] = $name;
	        }
	    }
		return $moduleList;
	}

	/*
	  * Return installed modules
	  *
	  * @param int $position Take only positionnables modules
	  * @return array Modules
	  */
	public static function getModulesInstalled($position = 0)
	{
		return Db::getInstance()->ExecuteS('
		SELECT *
		FROM `'._DB_PREFIX_.'module` m
		'.($position ? '
		LEFT JOIN `'._DB_PREFIX_.'hook_module` hm ON m.`id_module` = hm.`id_module`
		LEFT JOIN `'._DB_PREFIX_.'hook` k ON hm.`id_hook` = k.`id_hook`
		WHERE k.`position` = 1' : ''));
	}

	/*
	 * Execute modules for specified hook
	 *
	 * @param string $hook_name Hook Name
	 * @param array $hookArgs Parameters for the functions
	 * @return string modules output
	 */
	public static function hookExec($hook_name, $hookArgs = array(), $id_module = NULL)
	{
		if ((!empty($id_module) AND !Validate::isUnsignedId($id_module)) OR !Validate::isHookName($hook_name))
			die(Tools::displayError());

	 	global $cart, $cookie;

		if (get_class($cookie) == 'Cookie')
	 		$hookArgs['cookie'] = $cookie;
		if (get_class($cart) == 'Cart')
			$hookArgs['cart'] = $cart;

		$result = Db::getInstance()->ExecuteS('
    	SELECT h.`id_hook`, m.`name`, hm.`position`
    	FROM `'._DB_PREFIX_.'module` m
    	LEFT JOIN `'._DB_PREFIX_.'hook_module` hm ON hm.`id_module` = m.`id_module`
    	LEFT JOIN `'._DB_PREFIX_.'hook` h ON hm.`id_hook` = h.`id_hook`
    	WHERE h.`name` = \''.pSQL($hook_name).'\'
    	AND m.`active` = 1
    	'.($id_module ? 'AND m.`id_module` = '.intval($id_module) : '').'
			ORDER BY hm.`position`, m.`name` DESC');
		if (!$result)
			return false;
		$output = '';
		foreach ($result AS $k => $module)
    	{
			$moduleInstance = Module::getInstanceByName($module['name']);
			if (!$moduleInstance)
				return false;
			$exceptions = $moduleInstance->getExceptions($module['id_hook'], $moduleInstance->id);
			$fileindex = basename($_SERVER['PHP_SELF']);
			$show = true;
			if (!empty($exceptions) AND is_array($exceptions))
				foreach ($exceptions as $exception)
					if ($fileindex == $exception['file_name'])
						$show = false;
			if (is_callable(array($moduleInstance, 'hook'.$hook_name)) && $show)
				$output .= call_user_func(array($moduleInstance, 'hook'.$hook_name), $hookArgs);
		}
        return $output;
    }
    
    /*
	 * Get translation for a given module text
	 *
	 * @param string $string String to translate
	 * @return string Translation
	 */
	protected function l($string)
	{
		global $_MODULES, $_MODULE, $cookie;

		if (@include_once(_PS_MODULE_DIR_.$this->name.'/'.Language::getIsoById($cookie->id_lang).'.php'))
			$_MODULES = !empty($_MODULES) ? array_merge($_MODULES, $_MODULE) : $_MODULE;

		if (!is_array($_MODULES))
			return (str_replace('"', '&quot;', $string));

		$currentKey = '<{'.$this->name.'}'._THEME_NAME_.'>'.$this->page.'_'.md5(addslashes($string));
		$defaultKey = '<{'.$this->name.'}default>'.$this->page.'_'.md5(addslashes($string));

		return str_replace('"', '&quot;', (key_exists($currentKey, $_MODULES) ? stripslashes($_MODULES[$currentKey]) : (key_exists($defaultKey, $_MODULES) ? stripslashes($_MODULES[$defaultKey]) : $string)));
	}
	
    /*
	 * Reposition module
	 *
	 * @param boolean $id_hook Hook ID
	 * @param boolean $direction Up (1)  or Down (0)
	 */
	public function updatePosition($id_hook, $direction)
	{
		$position = intval($this->getPosition(intval($id_hook)));
		if ($position <= 1 AND !$direction)
			return false;
		
		Db::getInstance()->Execute('
		UPDATE `'._DB_PREFIX_.'hook_module`
		SET `position` = `position`'.($direction ? '-1' : '+1').'
		WHERE `id_hook` = '.intval($id_hook).'
		AND `position` = '.($direction ? $position + 1 : $position - 1));
		Db::getInstance()->Execute('
		UPDATE `'._DB_PREFIX_.'hook_module`
		SET `position` = `position`'.($direction ? '+1' : '-1').'
		WHERE `id_hook` = '.intval($id_hook).'
		AND `id_module` = '.intval($this->id));
		$this->cleanPositions($id_hook);
	}
	
    /*
	 * Reorder modules position
	 *
	 * @param boolean $id_hook Hook ID
	 */
	public function cleanPositions($id_hook)
	{
		$result = Db::getInstance()->ExecuteS('
		SELECT `id_module` 
		FROM `'._DB_PREFIX_.'hook_module`
		WHERE `id_hook` = '.intval($id_hook).'
		ORDER BY `position`');
		$sizeof = sizeof($result);
		for ($i = 0; $i < $sizeof; ++$i)
			Db::getInstance()->Execute('
			UPDATE `'._DB_PREFIX_.'hook_module`
			SET `position` = '.($i + 1).'
			WHERE `id_hook` = '.intval($id_hook).'
			AND `id_module` = '.intval($result[$i]['id_module']));
		return true;
	}
	
	
    /*
	 * Return module position for a given hook
	 *
	 * @param boolean $id_hook Hook ID
	 * @return integer position
	 */
	public function getPosition($id_hook)
	{		
		$result = Db::getInstance()->getRow('
		SELECT `position`
		FROM `'._DB_PREFIX_.'hook_module`
		WHERE `id_hook` = '.intval($id_hook).'
		AND `id_module` = '.intval($this->id));
		return $result['position'];
	}
	
	public function displayError($error)
	{
	 	$output = '
		<div class="module_error alert error">
			<img src="'.$this->_path.'warning.gif" alt="" title="" /> '.$error.'
		</div>';
		$this->error = true;
		
		return $output;
	}
	
	public function displayConfirmation($string)
	{
	 	$output = '
		<div class="module_confirmation conf confirm">
			<img src="'.$this->_path.'ok.gif" alt="" title="" /> '.$string.'
		</div>';
		
		return $output;
	}
	
	
    /*
	 * Return execeptions for module in hook
	 *
	 * @param int $id_hook Hook ID
	 * @return array Exceptions
	 */	
	public function getExceptions($id_hook)
	{
		return Db::getInstance()->ExecuteS('
		SELECT `file_name` 
		FROM `'._DB_PREFIX_.'hook_module_exceptions`
		WHERE `id_hook` = '.intval($id_hook).'
		AND  `id_module` = '.intval($this->id));
	}

	public static function display($file, $template)
	{
		global $smarty;

        $previousTemplate = $smarty->currentTemplate;
		$smarty->currentTemplate = substr(basename($template), 0, -4);
		if (file_exists(_PS_THEME_DIR_.'modules/'.basename($file, '.php').'/'.$template))
		{
			$smarty->assign('module_dir', _THEME_DIR_.'modules/'.basename($file, '.php').'/');
			$result = $smarty->fetch(_PS_THEME_DIR_.'modules/'.basename($file, '.php').'/'.$template);
		}
		elseif (file_exists(dirname($file).'/'.$template))
		{
			$smarty->assign('module_dir', __PS_BASE_URI__.'modules/'.basename($file, '.php').'/');
			$result = $smarty->fetch(dirname($file).'/'.$template);
		}
		else
			$result = '';
		$smarty->currentTemplate = $previousTemplate;
		return $result;
	}

}
