<?php

/**
  * PDF class, PDF.php
  * PDF invoices and document management
  * @category classes
  *
  * @author PrestaShop <support@prestashop.com>
  * @copyright PrestaShop
  * @license http://www.opensource.org/licenses/osl-3.0.php Open-source licence 3.0
  * @version 0.8
  *
  */

include_once(_PS_FPDF_PATH_.'fpdf.php');

class PDF extends FPDF
{
 	/** @var object Customer Order object */
	private static $order = NULL;

	/** @var object Order currency object */
	private static $currency = NULL;

	private static $_iso;
	/** @var array Translations chosen for display invoice */
	private static $_translations = array (
		'FR' => array(
			'INVOICE #' => 'FACTURE N°FA',
			'Invoice' => 'Facture',
			'from' => 'du',
			'An electronic version of this invoice is kept in Your Account. To access it, log in to the website' => 'Une version électronique de cette facture est archivée sur votre espace personnel. Pour y accéder, connectez-vous sur le site',
			'using your e-mail address and password (which you created while placing your first order).' => 'à l\'aide de votre adresse e-mail et de votre mot de passe (renseigné lors de votre première commande).',
			'Product' => 'Produit',
			'Products' => 'Produits',
			'Description' => 'Désignation',
			'Qty' => 'Qté',
			'Ref' => 'Réf',
			'Reference' => 'Référence',
			'Tax' => 'Taxe',
			'Price' => 'Prix HT',
			'U. price' => 'Prix U.',
			'Price TI' => 'Prix TTC',
			'Total TI' => 'Total TTC',
			'Discount:' => 'Réduction :',
			'Total products TI' => 'Total produits TTC',
			'Total discounts' => 'Total bons d\'achat et réductions',
			'Total shipping' => 'Frais d\'expédition',
			'Delivery' => 'Livraison',
			'Invoicing' => 'Facturation',
			'Order date:' => 'Date de commande :',
			'Shipping date:' => 'Date d\'expédition :',
			'Payment method:' => 'Mode de paiement :',
			'Carrier:' => 'Transporteur :',
			'Headquarters:' => 'Siège social :',
			'RCS:' => 'RCS :',
			'SIRET:' => 'SIRET :',
			'Tax detail' => 'Détail TVA',
			'Tax %' => 'TVA %',
			'Pre-Tax Total' => 'Total HT',
			'Total Tax' => 'Total TVA',
			'Total with Tax' => 'Total TTC',
			'PHONE:' => 'Tel. :',
			'Origin:' => 'Origine :',
		)
	);
	
	/**
	* Invoice header
	*/
	function Header()
	{
		$conf = Configuration::getMultiple(array('PS_SHOP_NAME', 'PS_SHOP_ADDR1', 'PS_SHOP_CODE', 'PS_SHOP_CITY', 'PS_SHOP_COUNTRY'));
		$conf['PS_SHOP_NAME'] = isset($conf['PS_SHOP_NAME']) ? utf8_decode($conf['PS_SHOP_NAME']) : 'Your company';		
		$conf['PS_SHOP_ADDR1'] = isset($conf['PS_SHOP_ADDR1']) ? utf8_decode($conf['PS_SHOP_ADDR1']) : 'Your company';
		$conf['PS_SHOP_CODE'] = isset($conf['PS_SHOP_CODE']) ? utf8_decode($conf['PS_SHOP_CODE']) : 'Postcode';
		$conf['PS_SHOP_CITY'] = isset($conf['PS_SHOP_CITY']) ? utf8_decode($conf['PS_SHOP_CITY']) : 'City';
		$conf['PS_SHOP_COUNTRY'] = isset($conf['PS_SHOP_COUNTRY']) ? utf8_decode($conf['PS_SHOP_COUNTRY']) : 'Country';

		if (file_exists(_PS_IMG_DIR_.'/logo.jpg'))
			$this->Image(_PS_IMG_DIR_.'/logo.jpg', 10, 8, 0, 15);

		$this->SetFont('Arial', 'B', 15);
		$this->Cell(115);
		$this->Cell(80, 10, self::l('INVOICE #').sprintf('%06d', self::$order->id), 0, 0, 'C');
	}

	/**
	* Invoice footer
	*/
	function Footer()
	{
		$this->SetY(-26);

		/*
		 * Display a message for customer
		 */
		$this->SetFont('Arial', '', 8);
		$this->Cell(0, 10, self::l('An electronic version of this invoice is kept in Your Account. To access it, log in to the website'), 0, 0, 'C', 0, 'http://'.$_SERVER['SERVER_NAME'].__PS_BASE_URI__.'history.php');
		$this->Ln(4);
		$this->Cell(0, 10, utf8_decode(Configuration::get('PS_SHOP_NAME')).' '.self::l('using your e-mail address and password (which you created while placing your first order).'), 0, 0, 'C', 0, 'http://'.$_SERVER['SERVER_NAME'].__PS_BASE_URI__.'history.php');
		$this->Ln(9);

		$arrayConf = array('PS_SHOP_NAME', 'PS_SHOP_ADDR1', 'PS_SHOP_CODE', 'PS_SHOP_CITY', 'PS_SHOP_COUNTRY', 'PS_SHOP_RCS', 'PS_SHOP_PHONE', 'PS_SHOP_SIRET');
		$conf = Configuration::getMultiple($arrayConf);
		$conf = array_map('utf8_decode', $conf);
		foreach ($arrayConf as $key)
			if (!isset($conf[$key]))
				$conf[$key] = '';
		$this->SetFillColor(240, 240, 240);
		$this->SetTextColor(0, 0, 0);
		$this->SetFont('Arial', '', 8);
		$this->Cell(0, 5, strtoupper($conf['PS_SHOP_NAME']).
		(!empty($conf['PS_SHOP_ADDR1']) ? ' - '.self::l('Headquarters:').' '.$conf['PS_SHOP_ADDR1'].(!empty($conf['PS_SHOP_ADDR2']) ? ' '.$conf['PS_SHOP_ADDR2'] : '').' '.$conf['PS_SHOP_CODE'].' '.$conf['PS_SHOP_CITY'].' '.$conf['PS_SHOP_COUNTRY'] : ''), 0, 1, 'C', 1);
		$this->Cell(0, 5, 
		(!empty($conf['PS_SHOP_RCS']) ? self::l('RCS:').' '.$conf['PS_SHOP_RCS'].' - ' : '').
		(!empty($conf['PS_SHOP_SIRET']) ? self::l('SIRET:').' '.$conf['PS_SHOP_SIRET'].' - ' : '').
		(!empty($conf['PS_SHOP_PHONE']) ? self::l('PHONE:').' '.$conf['PS_SHOP_PHONE'] : ''), 0, 1, 'C', 1);
	}

	/**
	* Main
	*
	* @param object $order Order
	* @param string $mode Download or display (optional)
	*/
	public static function invoice($order, $mode = 'D')
	{
	 	global $cookie;

		if (!Validate::isLoadedObject($order) OR (!isset($cookie->id_employee) AND (!OrderState::invoiceAvailable($order->getCurrentState()))))
			return false;
		self::$order = $order;
		self::$_iso = strtoupper(Language::getIsoById(intval(self::$order->id_lang)));
		self::$currency = new Currency(intval(self::$order->id_currency));
		
		$pdf = new PDF('P', 'mm', 'A4');
		$pdf->AliasNbPages();
		$pdf->AddPage();

		$invoice_address = new Address(intval($order->id_address_invoice));
		$delivery_address = new Address(intval($order->id_address_delivery));
		$shop_country = Configuration::get('PS_SHOP_COUNTRY');

		$width = 100;

		$pdf->SetX(10);
		$pdf->SetY(25);
		$pdf->SetFont('Arial', '', 12);
		$pdf->Cell($width, 10, self::l('Delivery'), 0, 'L');
		$pdf->Cell($width, 10, self::l('Invoicing'), 0, 'L');
		$pdf->Ln(5);
		$pdf->SetFont('Arial', '', 9);

		if (!empty($delivery_address->company) OR !empty($invoice_address->company))
		{
			$pdf->Cell($width, 10, utf8_decode($delivery_address->company), 0, 'L');
			$pdf->Cell($width, 10, utf8_decode($invoice_address->company), 0, 'L');
			$pdf->Ln(5);
		}
		$pdf->Cell($width, 10, utf8_decode($delivery_address->surname).' '.utf8_decode($delivery_address->name), 0, 'L');
		$pdf->Cell($width, 10, utf8_decode($invoice_address->surname).' '.utf8_decode($invoice_address->name), 0, 'L');
		$pdf->Ln(5);
		$pdf->Cell($width, 10, utf8_decode($delivery_address->address1), 0, 'L');
		$pdf->Cell($width, 10, utf8_decode($invoice_address->address1), 0, 'L');
		$pdf->Ln(5);
		if (!empty($invoice_address->address2) OR !empty($delivery_address->address2))
		{
			$pdf->Cell($width, 10, utf8_decode($delivery_address->address2), 0, 'L');
			$pdf->Cell($width, 10, utf8_decode($invoice_address->address2), 0, 'L');
			$pdf->Ln(5);
		}
		$pdf->Cell($width, 10, $delivery_address->postcode.' '.utf8_decode($delivery_address->city), 0, 'L');
		$pdf->Cell($width, 10, $invoice_address->postcode.' '.utf8_decode($invoice_address->city), 0, 'L');
		$pdf->Ln(5);
		$pdf->Cell($width, 10, utf8_decode($delivery_address->country), 0, 'L');
		$pdf->Cell($width, 10, utf8_decode($invoice_address->country), 0, 'L');
		$pdf->Ln(5);
		$pdf->Cell($width, 10, $delivery_address->phone, 0, 'L');
		if (!empty($delivery_address->phone_mobile))
		{
			$pdf->Ln(5);
			$pdf->Cell($width, 10, $delivery_address->phone_mobile, 0, 'L');
		}

		/*
		 * display order information
		 */
		$carrier = new Carrier(self::$order->id_carrier);
		$history = self::$order->getHistory(self::$order->id_lang);
		foreach($history as $h)
			if ($h['id_order_state'] == _PS_OS_SHIPPING_)
				$shipping_date = $h['date_add'];
		$pdf->Ln(12);		
		$pdf->SetFillColor(240, 240, 240);
		$pdf->SetTextColor(0, 0, 0);
		$pdf->SetFont('Arial', '', 9);
		$pdf->Cell(0, 6, self::l('INVOICE #').sprintf('%06d', self::$order->id).' '.self::l('from') . ' ' .Tools::displayDate(self::$order->date_upd, self::$order->id_lang), 1, 2, 'L', 1);
		$pdf->Cell(75, 6, self::l('Order date:').' '.Tools::displayDate(self::$order->date_add, self::$order->id_lang), 'L', 0);
		$pdf->Cell(50, 6, self::l('Carrier:').' '.utf8_decode($carrier->name), 'L');
		$pdf->Cell(0, 6, self::l('Payment method:').' '.utf8_decode($order->payment), 'LR');
		$pdf->Ln(5);
		$pdf->Cell(75, 6, (isset($shipping_date) ? self::l('Shipping date:').' '.Tools::displayDate($shipping_date, self::$order->id_lang) : ' '), 'LB', 0);
		$pdf->Cell(50, 6, ($shop_country ? self::l('Origin:').' '.utf8_decode($shop_country) : ' '), 'LRB');
		$pdf->Cell(0, 6, ' ', 'LRB');
		$pdf->Ln(15);
		$pdf->ProdTab();
		$pdf->DiscTab();

		/*
		 * Display price summation
		 */
		$pdf->Ln(5);
		$pdf->SetFont('Arial', 'B', 8);
		$width = 165;
		$pdf->Cell($width, 0, self::l('Total products TI').' : ', 0, 0, 'R');
		$pdf->Cell(0, 0, self::euro(Tools::displayPrice(self::$order->getTotalProductsWithTaxes(), self::$currency, true)), 0, 0, 'R');
		$pdf->Ln(4);
		
		$pdf->Cell($width, 0, self::l('Total discounts').' : ', 0, 0, 'R');
		$pdf->Cell(0, 0, self::euro(Tools::displayPrice(self::$order->total_discounts, self::$currency, true)), 0, 0, 'R');
		$pdf->Ln(4);
		
		$pdf->Cell($width, 0, self::l('Total shipping').' : ', 0, 0, 'R');
		$pdf->Cell(0, 0, self::euro(Tools::displayPrice(self::$order->total_shipping, self::$currency, true)), 0, 0, 'R');
		$pdf->Ln(4);
		
		$pdf->Cell($width, 0, self::l('Total with Tax').' : ', 0, 0, 'R');
		$pdf->Cell(0, 0, self::euro(Tools::displayPrice(self::$order->total_paid, self::$currency, true, false)), 0, 0, 'R');
		$pdf->Ln(5);

		$pdf->TaxTab();

		Hook::PDFInvoice($pdf, self::$order->id);

		return $pdf->Output(sprintf('%06d', self::$order->id).'.pdf', $mode);
	}

	/**
	* Product table with price, quantities...
	*/
	function ProdTab()
	{
		
		$header = array(
			array(self::l('Description'), 'L'),
			array(self::l('Reference'), 'L'),
			array(self::l('U. price'), 'R'),
			array(self::l('Qty'), 'C'),
			array(self::l('Pre-Tax Total'), 'R'), 
			array(self::l('Total'), 'R')
		);
		$w = array(90, 15, 25, 10, 25, 25);
		$this->SetFont('Arial', 'B', 8);
		$this->SetFillColor(240, 240, 240);
		for($i = 0; $i < count($header); $i++)
			$this->Cell($w[$i], 5, $header[$i][0], 'T', 0, $header[$i][1], 1);
		$this->Ln();
		$this->SetFont('Arial', '', 7);

		$products = self::$order->getProducts();
		foreach($products AS $product)
		{
			$unit_without_tax = $product['product_price'];
			$total_without_tax = $product['total_price'];
			$total_with_tax = $product['total_wt'];

			$before = $this->GetY();
            $this->MultiCell($w[0], 5, utf8_decode($product['product_name']), 'B');
			$lineSize = $this->GetY() - $before;
			$this->SetXY($this->GetX() + $w[0], $this->GetY() - $lineSize);
			$this->Cell($w[1], $lineSize, $product['product_reference'], 'B');
			$this->Cell($w[2], $lineSize, self::euro(Tools::displayPrice($unit_without_tax, self::$currency, true)), 'B', 0, 'R');
			$this->Cell($w[3], $lineSize, $product['product_quantity'], 'B', 0, 'C');
			$this->Cell($w[4], $lineSize, self::euro(Tools::displayPrice($total_without_tax, self::$currency, true)), 'B', 0, 'R');
			$this->Cell($w[5], $lineSize, self::euro(Tools::displayPrice($total_with_tax, self::$currency, true)), 'B', 0, 'R');
			$this->Ln();
		}

		if (!sizeof(self::$order->getDiscounts()))
			$this->Cell(array_sum($w), 0, '');
	}

	/**
	* Discount table with value, quantities...
	*/
	function DiscTab()
	{
		$w = array(90, 25, 15, 10, 25, 25);
		$this->SetFont('Arial', 'B', 7);
		$discounts = self::$order->getDiscounts();
		foreach($discounts AS $discount)
		{
			$this->Cell($w[0], 6, self::l('Discount:').$discount['name'], 'B');
			$this->Cell($w[1], 6, '', 'B');
			$this->Cell($w[2], 6, '', 'B');
			$this->Cell($w[3], 6, '1', 'B', 0, 'C');
			$this->Cell($w[4], 6, '', 'B', 0, 'R');
			$this->Cell($w[6], 6, '-'.self::euro(Tools::displayPrice($discount['value'], self::$currency, true)), 'B', 0, 'R');
			$this->Ln();
		}
		
		if (sizeof($discounts))
			$this->Cell(array_sum($w), 0, '');
	}

	/**
	* Tax table
	*/
	function TaxTab()
	{
		$header = array(self::l('Tax detail'), self::l('Tax %'), self::l('Pre-Tax Total'), self::l('Total Tax'), self::l('Total with Tax'));
		$w = array(60, 30, 40, 30, 30);
		$this->SetFont('Arial', 'B', 8);
		for($i = 0; $i < count($header); $i++)
		{
			$this->Cell($w[$i], 5, $header[$i], 0, 0, 'R');
		}

		$this->Ln();
		$this->SetFont('Arial', '', 7);

		$products = self::$order->getProducts();
		$total_with_tax = array();
		$total_without_tax = array();
		$taxes = array();
		foreach ($products AS $product)
		{
			if (!isset($total[$product['tax_rate']]))
				$total_with_tax[$product['tax_rate']] = 0;
			if (!isset($total[$product['tax_rate']]))
				$total_without_tax[$product['tax_rate']] = 0;
			if (!isset($taxes[$product['tax_rate']]))
				$taxes[$product['tax_rate']] = 0;
				
			$price_with_tax = number_format($product['product_price'] * (1 + ($product['tax_rate'] / 100)), 2, '.', '');
			$price_without_tax = $product['product_price'];
			$vat = ($price_with_tax - $price_without_tax) * $product['product_quantity'];
			
			$taxes[$product['tax_rate']] += $vat;
			$total_with_tax[$product['tax_rate']] += $price_with_tax * $product['product_quantity'];
			$total_without_tax[$product['tax_rate']] += $price_without_tax * $product['product_quantity'];
		}
		if (intval(Configuration::get('PS_TAX_NO')))
		{
			foreach ($taxes AS $tax_rate => $vat)
			{
				if ($tax_rate == "0.00")
					continue ;
				$before = $this->GetY();
				$lineSize = $this->GetY() - $before;
				$this->SetXY($this->GetX(), $this->GetY() - $lineSize + 3);
				$this->Cell($w[0], $lineSize, self::l('Products'), 0, 0, 'R');
				$this->Cell($w[1], $lineSize, number_format($tax_rate, 2, ',', ' '), 0, 0, 'R');
				$this->Cell($w[2], $lineSize, self::euro(Tools::displayPrice($total_without_tax[$tax_rate], self::$currency, true)), 0, 0, 'R');
				$this->Cell($w[3], $lineSize, self::euro(Tools::displayPrice($vat, self::$currency, true)), 0, 0, 'R');
				$this->Cell($w[4], $lineSize, self::euro(Tools::displayPrice($total_with_tax[$tax_rate], self::$currency, true)), 0, 0, 'R');
				$this->Ln();
			}
		}
		$carrier = new Carrier(self::$order->id_carrier);
		
		$carrier_taxe = new Tax($carrier->id_tax);
		$total_shipping_wt = self::$order->total_shipping / (1 + ($carrier_taxe->rate / 100));
		
		$before = $this->GetY();
		$lineSize = $this->GetY() - $before;
		$this->SetXY($this->GetX(), $this->GetY() - $lineSize + 3);
		$this->Cell($w[0], $lineSize, self::l('Carrier'), 0, 0, 'R');
		$this->Cell($w[1], $lineSize, number_format($carrier_taxe->rate, 2, ',', ' '), 0, 0, 'R');
		$this->Cell($w[2], $lineSize, self::euro(Tools::displayPrice($total_shipping_wt, self::$currency, true)), 0, 0, 'R');
		$this->Cell($w[3], $lineSize, self::euro(Tools::displayPrice(self::$order->total_shipping - $total_shipping_wt, self::$currency, true)), 0, 0, 'R');
		$this->Cell($w[4], $lineSize, self::euro(Tools::displayPrice(self::$order->total_shipping, self::$currency, true)), 0, 0, 'R');
		$this->Ln();
	}

	static private function euro($s)
	{
		return str_replace('&euro;', chr(128), $s);
	}
	
	static private function l($key)
	{
		return (utf8_decode(Tools::historyc_l($key, self::$_translations)));
	}
}

?>