<?php

/**
  * Payment modules class, PaymentModule.php
  * Payment modules management and abstraction
  * @category classes
  *
  * @author PrestaShop <support@prestashop.com>
  * @copyright PrestaShop
  * @license http://www.opensource.org/licenses/osl-3.0.php Open-source licence 3.0
  * @version 0.9
  *
  */

include_once(dirname(__FILE__).'/../config/config.inc.php');

abstract class PaymentModule extends Module
{
	/** @var integer Current order's id */
	public	$currentOrder;

	/**
	* Validate an order in database
	* Function called from a payment module
	*
	* @param integer $id_cart Value
	* @param integer $id_order_state Value
	* @param float $amountPaid Amount really paid by customer
	* @param string $paymentMethod Payment method (eg. 'Credit cart')
	* @param string $message Message to attach to order
	*/

	function validateOrder($id_cart, $id_order_state, $amountPaid, $paymentMethod = 'Unknown', $message = NULL, $currency_special = NULL)
	{
		$cart = new Cart(intval($id_cart));

		/* Does order already exists ? */
		if (Validate::isLoadedObject($cart) AND $cart->OrderExists() == 0)
		{
			/* Copying data from cart */
			$order = new Order();
			$order->id_carrier = $cart->id_carrier;
			$order->id_customer = $cart->id_customer;
			$order->id_address_invoice = $cart->id_address_invoice;
			$order->id_address_delivery = $cart->id_address_delivery;
			$tva_address = new Address(intval($order->id_address_delivery));
			$tva_country = new Country(intval($tva_address->id_country));
			$order->id_currency = ($currency_special ? intval($currency_special) : $cart->id_currency);
			$order->id_lang = $cart->id_lang;
			$order->id_cart = $cart->id;
			$order->payment = substr($paymentMethod, 0, 32);
			if (isset($this->name))
				$order->module = $this->name;
			$order->total_paid_real = floatval($amountPaid);
			$order->recyclable = $cart->recyclable;
			$order->gift = $cart->gift;
			$order->gift_message = $cart->gift_message;
			$currency = new Currency($order->id_currency);
			$order->total_products = number_format($cart->getOrderTotal(false, 1), 2, '.', '');
			$order->total_discounts = number_format(abs($cart->getOrderTotal(true, 2)), 2, '.', '');
			$order->total_shipping = number_format($cart->getOrderShippingCost(), 2, '.', '');
			$order->total_paid = number_format(Tools::convertPrice($cart->getOrderTotal(true, 3), $currency), 2, '.', '');

			/* Amount paid by customer is not the right one -> Status = payment error */
			if ($order->total_paid != number_format($amountPaid, 2, '.', ''))
				$id_order_state = _PS_OS_ERROR_;

			/* Creating order */
			$result = $order->add();
			$result = true;

			/* Next ! */
			if ($result AND isset($order->id))
			{
				/* Set order state in order history */
				if ($id_order_state == _PS_OS_CHEQUE_)
					$data_history =  array(
					'{cheque_name}' => Configuration::get('CHEQUE_NAME'),
					'{cheque_address}' => Configuration::get('CHEQUE_ADDRESS'),
					'{cheque_address_html}' => nl2br(Configuration::get('CHEQUE_ADDRESS')));
				if ($id_order_state == _PS_OS_BANKWIRE_)
					$data_history =  array(
					'{bankwire_home}' => Configuration::get('BANK_WIRE_HOME'),
					'{bankwire_home_html}' => nl2br(Configuration::get('BANK_WIRE_HOME')),
					'{bankwire_owner}' => Configuration::get('BANK_WIRE_OWNER'),
					'{bankwire_iban}' => Configuration::get('BANK_WIRE_IBAN'),
					'{bankwire_bic}' => Configuration::get('BANK_WIRE_BIC'));
				$history = new OrderHistory();
				$history->id_order = intval($order->id);
				$history->changeIdOrderState(intval($id_order_state), intval($order->id));
				$history->addWithemail(true, (isset($data_history) ? $data_history : false));

				/* Optional message to attach to this order */
				if (isset($message) AND !empty($message))
				{
					$msg = new Message();
					$message = strip_tags($message, '<br>');
					if (!Validate::isMessage($message))
						$message = $this->l('Payment message is not valid, please check your module!');
					$msg->message = $message;
					$msg->id_order = intval($order->id);
					$msg->private = 1;
					$msg->add();
				}

				/* Insert products from cart into order_detail table */
				$products = $cart->getProducts();
				$productsList = '';
				$db = Db::getInstance();
				$query = 'INSERT INTO `'._DB_PREFIX_.'order_detail` 
					(`id_order`, `product_id`, `product_attribute_id`, `product_name`, `product_quantity`, `product_price`, `product_reference`, `product_weight`, `tax_name`, `tax_rate`, `download_deadline`, `download_hash`) 
				VALUES ';

				foreach ($products AS $product)
				{
					if ($id_order_state != _PS_OS_CANCELED_ AND $id_order_state != _PS_OS_ERROR_)
					{
						if (($id_order_state != _PS_OS_OUTOFSTOCK_) AND (($updateResult = Product::updateQuantity($product)) === false OR $updateResult === -1))
							{
								$id_order_state = _PS_OS_OUTOFSTOCK_;
								$history = new OrderHistory();
								$history->id_order = intval($order->id);
								$history->changeIdOrderState(_PS_OS_OUTOFSTOCK_);
								$history->addWithemail();
							}
						Hook::updateQuantity($product, $order);
					}
					$price = Product::getPriceStatic(intval($product['id_product']), false, ($product['id_product_attribute'] ? intval($product['id_product_attribute']) : NULL), 4);

					/* Add some informations for virtual products */
					$deadline = NULL;
					$download_hash = NULL;
					$productDownload = new ProductDownload();
					if ($id_product_download = $productDownload->getIdFromIdProduct(intval($product['id_product'])))
					{
						$productDownload = new ProductDownload($id_product_download);
						$deadline = $productDownload->getDeadLine();
						$download_hash = $productDownload->getHash();
					}
					
					/* Exclude TVA */
					if (Tax::excludeTaxeByZone($product['id_tax'], $tva_country) OR Tax::excludeTaxeOption())
					{
						$product['tax'] = 0;
						$product['rate'] = 0;
					}

					$query .= '('.intval($order->id).',
							'.intval($product['id_product']).',
							'.(isset($product['id_product_attribute']) ? intval($product['id_product_attribute']) : 'NULL').',
							\''.pSQL($product['name'].((isset($product['attributes']) AND $product['attributes'] != NULL) ? ' - '.$product['attributes'] : '')).'\',
							'.intval($product['quantity']).',
							'.$price.',
							\''.pSQL($product['reference']).'\',
							'.floatval($product['weight']).',
							\''.pSQL($product['tax']).'\',
							'.floatval($product['rate']).',
							\''.pSQL($deadline).'\',
							\''.pSQL($download_hash).'\'),';

					$productsList .=
						'<tr>
								<td align="center" style="height: 25px; border-left: 1px solid #000000;">'.intval($product['quantity']).'</td>
								<td style=" border-left: 1px solid #000000; border-right: 1px solid #000000;">'.$product['name'].(isset($product['attributes_small']) ? ' '.$product['attributes_small'] : '').'</td>
								<td align="right" style="border-right: 1px solid #000000;">'.Tools::displayPrice($price * ($product['rate'] + 100) / 100, $currency).'</td>
						</tr>';
				} // end foreach ($products)
				$query = rtrim($query, ',');
				$result = $db->Execute($query);

				/* Insert discounts from cart into order_discount table */
				$discounts = $cart->getDiscounts();
				$discountsList = '';
				foreach ($discounts AS $discount)
				{
					$objDiscount = new Discount($discount['id_discount']);
					$value = number_format($objDiscount->getValue(sizeof($discounts), $cart->getOrderTotal(true, 1), $order->total_shipping), 2);
					$order->addDiscount($objDiscount->id, $objDiscount->name, $value);
					if ($id_order_state != _PS_OS_ERROR_ AND $id_order_state != _PS_OS_CANCELED_)
						$objDiscount->quantity = $objDiscount->quantity - 1;
					$objDiscount->update();

					$discountsList .=
					'<tr>
							<td align="center" style="height: 25px; border-left: 1px solid #000000;">1</td>
							<td style=" border-left: 1px solid #000000; border-right: 1px solid #000000;">'.$this->l('Voucher code:').' '.$objDiscount->name.'</td>
							<td align="right" style="border-right: 1px solid #000000;">-'.Tools::displayPrice($value, $currency).'</td>
					</tr>';
				}

				/* Specify order id for message */
				$oldMessage = Message::getMessageByCartId(intval($cart->id));
				if ($oldMessage)
				{
					$message = new Message(intval($oldMessage['id_message']));
					$message->id_order = intval($order->id);
					$message->update();
				}

				/* Hook new order */
				$customer = new Customer($order->id_customer);
				$orderStatus = new OrderState(intval($id_order_state));
				if (Validate::isLoadedObject($orderStatus))
					Hook::newOrder($cart, $order, $customer, $currency, $orderStatus);

				/* Send an e-mail to customer */
				if ($id_order_state != _PS_OS_ERROR_ AND $id_order_state != _PS_OS_CANCELED_ AND $customer->id)
				{
					$invoice = new Address(intval($order->id_address_invoice));
					$delivery = new Address(intval($order->id_address_delivery));
					$carrier = new Carrier(intval($order->id_carrier));

					$data = array(
						'{surname}' => $customer->surname,
						'{name}' => $customer->name,
						'{email}' => $customer->email,				
						'{delivery_surname}' => $delivery->surname,
						'{delivery_name}' => $delivery->name,
						'{delivery_address1}' => $delivery->address1,
						'{delivery_address2}' => $delivery->address2,
						'{delivery_city}' => $delivery->city,
						'{delivery_postal_code}' => $delivery->postcode,
						'{delivery_country}' => $delivery->country,
						'{invoice_surname}' => $invoice->surname,
						'{invoice_name}' => $invoice->name,
						'{invoice_address2}' => $invoice->address2,
						'{invoice_address1}' => $invoice->address1,
						'{invoice_city}' => $invoice->city,
						'{invoice_postal_code}' => $invoice->postcode,
						'{invoice_country}' => $invoice->country,
						'{order_name}' => sprintf("%06d", intval($order->id)),
						'{date}' => Tools::displayDate(date('Y-m-d h:i:s'), intval($order->id_lang), 1),
						'{carrier}' => $carrier->name,
						'{payment}' => $order->payment,
						'{products}' => $productsList,
						'{discounts}' => $discountsList,
						'{total_paid}' => Tools::displayPrice($order->total_paid, $currency, false, false),
						'{total_products}' => Tools::displayPrice(($order->total_paid - Tools::convertPrice($order->total_shipping, $currency) + Tools::convertPrice($order->total_discounts, $currency)), $currency, false, false),
						'{total_discounts}' => Tools::displayPrice($order->total_discounts, $currency),
						'{total_shipping}' => Tools::displayPrice($order->total_shipping, $currency)
					);
					
					/* Join PDF invoice */
					
					if (Validate::isLoadedObject($orderStatus) AND $orderStatus->invoice)
					{
						$fileAttachment['content'] = PDF::invoice($order, 'S');
						$fileAttachment['name'] = sprintf("FA%06d.pdf", intval($order->id));
						$fileAttachment['mime'] = 'application/pdf';
					}
					else
						$fileAttachment = NULL;
					
					if (Validate::isemail($customer->email))
						Mail::Send(intval($order->id_lang), 'order_conf', 'Order confirmation', $data, 
						$customer->email, $customer->surname.' '.$customer->name, NULL, NULL, $fileAttachment);

					$this->currentOrder = $order->id;
					return true;
				}
			}
			else
			{
				die(Tools::displayError('Order creation failed'));
			}
		}
		else
		{
			die(Tools::displayError('An order has already been placed using this cart'));
		}
	}

}

?>