<?php

/**
  * Tools class, Tools.php
  * Various tools
  * @category classes
  *
  * @author PrestaShop <support@prestashop.com>
  * @copyright PrestaShop
  * @license http://www.opensource.org/licenses/osl-3.0.php Open-source licence 3.0
  * @version 0.8
  *
  */

class Tools
{
 	/**
	* Random password generator
	*
	* @param integer $length Desired length (optional)
	* @return string Password
	*/
	static public function passwdGen($length = 8)
	{
		$str = 'abcdefghijkmnopqrstuvwxyz0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ.';
    	srand((double)microtime() * 1234567);
    	for ($i = 0, $passwd = ''; $i < $length; $i++)
        	$passwd .= substr($str, (rand() % strlen($str)), 1);
    	return $passwd;
	}
	
	/**
	* Redirect user to another page
	*
	* @param string $url Desired URL
	* @param string $baseUri Base URI (optional)
	*/
	static public function redirect($url, $baseUri = __PS_BASE_URI__)
	{
		if (isset($_SERVER['HTTP_REFERER']) AND ($url == $_SERVER['HTTP_REFERER']))
			header('Location: '.$_SERVER['HTTP_REFERER']);
		else
			header('Location: '.$baseUri.$url);
		exit;
	}
	
	/**
	* Redirect url wich allready PS_BASE_URI
	*
	* @param string $url Desired URL
	*/
	static public function redirectLink($url)
	{
		header('Location: '.$url);
		exit;
	}
	
	/**
	* Redirect user to another admin page
	*
	* @param string $url Desired URL
	*/
	static public function redirectAdmin($url)
	{
		header('Location: '.$url);
		exit;
	}
	
	/**
	* Get a value from $_POST / $_GET
	* if unavailable, take a default value
	*
	* @param string $key Value key
	* @param mixed $defaultValue (optional)
	* @return mixed Value
	*/
	static public function getValue($key, $defaultValue = false)
	{
	 	if (!isset($key) OR empty($key) OR !is_string($key)) return false;
	 	$ret = isset($_POST[$key]) ? $_POST[$key] : (isset($_GET[$key]) ? $_GET[$key] : $defaultValue);
		return is_array($ret) ? $ret : stripslashes($ret);
	}
	
	/**
	* Change language in cookie while clicking on a flag
	*/
	static public function setCookieLanguage()
	{
		global $cookie;

		if (get_class($cookie) != 'Cookie')
			die (Tools::displayError());
		
		/* Language switching */
		if ($id_lang = intval(Tools::getValue('id_lang')) AND Validate::isUnsignedId($id_lang))
		{
			$cookie->id_lang = $id_lang;
			$link = new Link();
			if ($id_product = intval(Tools::getValue('id_product')) AND !isset($_GET['adminlang']))
				$url = $link->getProductLink(new Product($id_product, false, $id_lang));
			elseif ($id_category = intval(Tools::getValue('id_category')) AND !isset($_GET['adminlang']))
				$url = $link->getCategoryLink(new Category($id_category, $id_lang));
			else
			{
				$n = 0;
				$url = $_SERVER['PHP_SELF'];
				unset($_GET['id_lang'], $_GET['adminlang']);
				foreach ($_GET AS $k => $value)
					$url .= ((!$n++) ? '?' : '&').urlencode(stripslashes($k)).'='.urlencode(stripslashes($value));
			}
			Tools::redirectLink($url);
		}

		/* Automatically detect language if not already defined */
		if (!isset($cookie->id_lang) AND isset($_SERVER['HTTP_ACCEPT_LANGUAGE']))
		{
			$array = split(',', strtolower($_SERVER['HTTP_ACCEPT_LANGUAGE']));
			if (Validate::isLanguageIsoCode($array[0]))
				$cookie->id_lang = Language::getIdByIso($array[0]);
		}
		/* If language file not present, you must use default language file */
		if (!isset($cookie->id_lang) OR !Validate::isUnsignedId($cookie->id_lang))
			$cookie->id_lang = Configuration::get('PS_LANG_DEFAULT');
		
		$iso = Language::getIsoById($cookie->id_lang);
		@include_once(_PS_TRANSLATIONS_DIR_.$iso.'/fields.php');
		@include_once(_PS_TRANSLATIONS_DIR_.$iso.'/errors.php');
		@include_once(_PS_THEME_DIR_.'lang/'.$iso.'.php');
		return $iso;
	}
	
	static public function setCurrency()
	{
		global $cookie;
		
		if (get_class($cookie) != 'Cookie')
			die (Tools::displayError());

		if (isset($_POST['SubmitCurrency']) OR isset($_POST['SubmitCurrency_x']) OR isset($_POST['SubmitCurrency_y']))
			if (isset($_POST['id_currency']) AND is_numeric($_POST['id_currency']))
			{
				$currency = new Currency(intval($_POST['id_currency']));
				if (is_object($currency) AND $currency->id AND !$currency->deleted)
					$cookie->id_currency = intval($currency->id);
			}
	
		if (isset($cookie->id_currency))
		{
			$currency = new Currency(intval($cookie->id_currency));
			if (is_object($currency) AND $currency->id AND intval($currency->deleted) != 1)
				return $currency;
		}
		$currency = new Currency(intval(Configuration::get('PS_CURRENCY_DEFAULT')));
		if (is_object($currency) AND $currency->id)
			$cookie->id_currency = intval($currency->id);
		return $currency;
	}
	
	/**
	* Return price with currency sign for a given product
	*
	* @param float $price Product price
	* @param object $currency Current currency object
	* @param boolean $convert Need to convert currency sign to UTF8 (for FPDF), (optional)
	* @return string Price with currency sign
	*/
	static public function displayPrice($price, $currency, $no_utf8 = false, $convert = true)
	{
		if ($convert)
			$price = self::convertPrice($price, $currency);
		$c_sign = (is_array($currency) ? $currency['sign'] : $currency->sign);
		$c_format = (is_array($currency) ? $currency['format'] : $currency->format);
		switch ($c_format)
	 	{
	 	 	/* Currencies like $ -> $ 1,200 */
	 	 	case 1:
				$ret = $c_sign.' '.number_format($price, 2, '.', ',');
				break;

			/* Currencies like ¤ -> 1200 ¤ */
			case 2:
			default:
				$ret = number_format($price, 2, ',', ' ').' '.$c_sign;
		}
		if ($no_utf8)
			return str_replace('€', chr(128), $ret);
		return $ret;
	}

	/**
	* Return price converted
	*
	* @param float $price Product price
	* @param object $currency Current currency object
	*/
	static public function convertPrice($price, $currency)
	{
		$c_id = (is_array($currency) ? $currency['id_currency'] : $currency->id);
		$c_rate = (is_array($currency) ? $currency['conversion_rate'] : $currency->conversion_rate);
		if ($c_id != intval(Configuration::get('PS_CURRENCY_DEFAULT')))
			$price *= $c_rate;
		return $price;
	}

	/**
	* Display date regarding to language preferences
	*
	* @param array $params Date, format...
	* @param object $smarty Smarty object for language preferences
	* @return string Date
	*/
	static public function dateFormat($params, &$smarty)
	{
		return Tools::displayDate($params['date'], $smarty->ps_language->id, $params['full']);
	}
	
	/**
	* Display date regarding to language preferences
	*
	* @param string $date Date to display
	* @param integer $id_lang Language id
	* @param boolean $full With time or not (optional)
	* @return string Date
	*/
	static public function displayDate($date, $id_lang, $full = false, $separator='-')
	{
	 	if (!$date OR !strtotime($date))
	 		return false;
		if (!Validate::isDate($date) OR !Validate::isBool($full))
			die (Tools::displayError('Invalid date'));
	 	$tmpTab = explode($separator, substr($date, 0, 10));
	 	$hour = ' '.substr($date, -8);

	 	if ($id_lang == 2)
	 		return ($tmpTab[2].'-'.$tmpTab[1].'-'.$tmpTab[0].($full ? $hour : ''));
	 	else
	 		return ($tmpTab[0].'-'.$tmpTab[1].'-'.$tmpTab[2].($full ? $hour : ''));
	}
	
	/**
	* Sanitize a string
	*
	* @param string $string String to sanitize
	* @param boolean $full String contains HTML or not (optional)
	* @return string Sanitized string
	*/
	static public function safeOutput($string, $html = false)
	{
	 	if (!$html)
			$string = @htmlentities(strip_tags($string), ENT_QUOTES, 'utf-8');
		return $string;
	}
	
	/**
	* Delete directory and subdirectories
	*
	* @param string $dirname Directory name
	*/
	static public function deleteDirectory($dirname)
	{
		$files = scandir($dirname);
		foreach ($files as $file)
			if ($file != '.' AND $file != '..')
			{
				if (is_dir($file))
					self::deleteDirectory($file);
				elseif (file_exists($file))
					unlink($file);
				else
					echo 'Unable to delete '.$file;				
			}
		rmdir($dirname);
	}
	
	/**
	* Display an error according to an error code
	*
	* @param integer $code Error code
	*/
	static public function displayError($key = 'Hack attempt')
	{
		global $_ERRORS;
		return (isset($_ERRORS) AND is_array($_ERRORS) AND key_exists(md5(pSQL($key)), $_ERRORS)) ? $_ERRORS[md5(pSQL($key))] : $key;
	}
	
	/**
	* Display an error with detailed object
	*
	* @param object $object Object to display
	*/
	static public function dieObject($object)
	{
		echo '<pre style="text-align: left;">';
		print_r($object);
		echo '</pre>';
		die('<br />END');
	}
	
	/**
	* Check if submit has been posted
	*
	* @param string $submit submit name
	*/
	static public function isSubmit($submit)
	{
		return (isset($_POST[$submit]) OR isset($_POST[$submit.'_x']) OR isset($_POST[$submit.'_y']));
	}
	
	/**
	* Get meta tages for a given page
	*
	* @param integer $id_lang Language id
	* @return array Meta tags
	*/
	static public function getMetaTags($id_lang)
	{
	 	/* Products specifics meta tags */
		if ($id_product = Tools::getValue('id_product'))
		{
			$row = Db::getInstance()->getRow('
			SELECT `name`, `meta_title`, `meta_description`, `meta_keywords` 
			FROM `'._DB_PREFIX_.'product_lang` 
			WHERE id_lang = '.intval($id_lang).' AND id_product = '.intval($id_product));
			if ($row)
				return self::completeMetaTags($row, $row['name']);
		}
		
		/* Categories specifics meta tags */
		elseif ($id_category = Tools::getValue('id_category'))
		{
			$row = Db::getInstance()->getRow('
			SELECT `name`, `meta_title`, `meta_description`, `meta_keywords` 
			FROM `'._DB_PREFIX_.'category_lang` 
			WHERE id_lang = '.intval($id_lang).' AND id_category = '.intval($id_category));
			if ($row)
				return self::completeMetaTags($row, Category::hideCategoryPosition($row['name']));
		}
	
		/* Default meta tags */
		return Tools::getHomeMetaTags($id_lang);
	}
	
	/**
	* Get meta tags for a given page
	*
	* @param integer $id_lang Language id
	* @return array Meta tags
	*/
	static public function getHomeMetaTags($id_lang)
	{
		GLOBAL $cookie;
		
		/* Metas-tags */
		$ret['meta_title'] = Configuration::get('PS_META_TITLE', intval($cookie->id_lang)) ? Configuration::get('PS_META_TITLE', intval($cookie->id_lang)) : Configuration::get('PS_SHOP_NAME');
		$ret['meta_description'] = Configuration::get('PS_META_DESCRIPTION', intval($cookie->id_lang)) ? Configuration::get('PS_META_DESCRIPTION', intval($cookie->id_lang)) : '';
		$ret['meta_keywords'] = Configuration::get('PS_META_KEYWORDS', intval($cookie->id_lang)) ? Configuration::get('PS_META_KEYWORDS', intval($cookie->id_lang)) : '';
		return $ret;
	}
	

	static public function completeMetaTags($metaTags, $defaultValue)
	{
		global $cookie;

		if ($metaTags['meta_title'] == NULL)
			$metaTags['meta_title'] = $defaultValue;
		if ($metaTags['meta_description'] == NULL)
			$metaTags['meta_description'] = Configuration::get('PS_META_DESCRIPTION', intval($cookie->id_lang)) ? Configuration::get('PS_META_DESCRIPTION', intval($cookie->id_lang)) : '';
		if ($metaTags['meta_keywords'] == NULL)
			$metaTags['meta_keywords'] = Configuration::get('PS_META_KEYWORDS', intval($cookie->id_lang)) ? Configuration::get('PS_META_KEYWORDS', intval($cookie->id_lang)) : '';
		return $metaTags;
	}

	/**
	* Encrypt password
	*
	* @param object $object Object to display
	*/
	static public function encrypt($passwd)
	{
		return md5(pSQL(_COOKIE_KEY_.$passwd));
	}
	
	/**
	* Get the user's journey
	*
	* @param integer category id
	* @param string finish of the path
	*/
	static public function getPath($id_category, $path = '')
	{
		global $link, $cookie;
		if (get_class($link) != 'Link' OR get_class($cookie) != 'Cookie')
			die (Tools::displayError());
		
		$category = new Category(intval($id_category), intval($cookie->id_lang));
		if (!Validate::isLoadedObject($category))
			die (Tools::displayError());
		$pipe = (Configuration::get('PS_NAVIGATION_PIPE') ? Configuration::get('PS_NAVIGATION_PIPE') : '>');
		if ($category->id == 1)
			return '<span class="navigation_end">'.$path.'</span>';
		$path = '<a href="'.Tools::safeOutput($link->getCategoryLink($category)).'">'.Tools::safeOutput(Category::hideCategoryPosition($category->name)).'</a> '.$pipe.' '.$path;
		return Tools::getPath(intval($category->id_parent), $path);
	}
	
	/**
	* Stats for admin panel
	*
	* @return integer Categories total
	*/
	
	static public function getCategoriesTotal()
	{
		$row = Db::getInstance()->getRow('SELECT COUNT(`id_category`) AS total FROM `'._DB_PREFIX_.'category`');
		return intval($row['total']);
	}
	
	/**
	* Stats for admin panel
	*
	* @return integer Products total
	*/
	
	static public function getProductsTotal()
	{
		$row = Db::getInstance()->getRow('SELECT COUNT(`id_product`) AS total FROM `'._DB_PREFIX_.'product`');
		return intval($row['total']);
	}
	
	/**
	* Stats for admin panel
	*
	* @return integer Customers total
	*/
	
	static public function getCustomersTotal()
	{
		$row = Db::getInstance()->getRow('SELECT COUNT(`id_customer`) AS total FROM `'._DB_PREFIX_.'customer`');
		return intval($row['total']);
	}
	
	/**
	* Stats for admin panel
	*
	* @return integer Orders total
	*/
	
	static public function getOrdersTotal()
	{
		$row = Db::getInstance()->getRow('SELECT COUNT(`id_order`) AS total FROM `'._DB_PREFIX_.'orders`');
		return intval($row['total']);
	}

	/*
	** Historyc translation function kept for compatibility
	** Removing soon
	*/
	static public function historyc_l($key, $translations)
	{
		global $cookie;
		if (!$translations OR !is_array($translations))
			die(Tools::displayError());
		$iso = strtoupper(Language::getIsoById($cookie->id_lang));
		$lang = key_exists($iso, $translations) ? $translations[$iso] : false;
		return (($lang AND is_array($lang) AND key_exists($key, $lang)) ? stripslashes($lang[$key]) : $key);
	}

	/*
	** Module translation function
	** MUST BE USED IN MODULES ONLY; AND ONLY WHERE TEMPLATES CANNOT DISPLAY STRINGS (e.g. errors)
	*/
	static public function l($string, $mod, $file, $js = false)
	{
		/*
		* Warning : 2 lines have been added to the Smarty class.
		* "public $currentTemplate = null;" into the class itself
		* "$this->currentTemplate = substr(basename($resource_name), 0, -4);" into the "display" method
		*/
		if (!$string OR empty($string) OR !$mod OR empty($mod) OR !$file OR empty($file))
			die(Tools::displayError());

		global $_MODULES, $_MODULE, $smarty, $cookie;
		$key = basename($file, '.php').'_'.md5(addslashes($string));
	
		if (file_exists(_PS_THEME_DIR_.'modules/'.$mod.'/'.Language::getIsoById($cookie->id_lang).'.php'))
		{
			$translationsFile = _PS_THEME_DIR_.'modules/'.$mod.'/'.Language::getIsoById($cookie->id_lang).'.php';
			$modKey = '<{'.$mod.'}'._THEME_NAME_.'>'.$key;
		}
		else
		{
			$translationsFile = _PS_MODULE_DIR_.$mod.'/'.Language::getIsoById($cookie->id_lang).'.php';
			$modKey = '<{'.$mod.'}default>'.$key;
		}

		if (@include_once($translationsFile))
			$_MODULES = array_merge($_MODULES, $_MODULE);

		$msg = (is_array($_MODULES) AND key_exists($modKey, $_MODULES)) ? ($js ? addslashes($_MODULES[$modKey]) : stripslashes($_MODULES[$modKey])) : $string;
		return str_replace('"', '&quot;', $msg);
	}
	
	static public function link_rewrite($str, $utf8_decode = false)
	{
	 	$purified = '';
	 	if ($utf8_decode)
	 		$str = utf8_decode($str);
	 	for ($i = 0; $i < strlen($str); $i++)
			if (strlen(htmlentities($str[$i])) > 1)
			{
				$entity = htmlentities($str[$i]);
				$purified .= $entity[1];
			}
	 	 	elseif (preg_match('|[[:alpha:]]{1}|', $str[$i]))
				$purified .= $str[$i];
			elseif (preg_match('<[[:digit:]]|-{1}>', $str[$i]))
				$purified .= $str{$i};
			elseif ($str[$i] == ' ')
				$purified .= '-';
		return $purified;
	}
	
	/**
	* Truncate strings
	*
	* @param string $str
	* @param integer $maxLen Max length
	* @param string $suffix Suffix optional
	* @return string $str truncated
	*/
	/* CAUTION : Use it only on module hookEvents.
	** For other purposes use the smarty function instead */
	static public function truncate($str, $maxLen, $suffix = '...')
	{
	 	if (strlen($str) <= $maxLen)
	 		return $str;
	 	$str = utf8_decode($str);
	 	return (utf8_encode(substr($str, 0, $maxLen - strlen($suffix)).$suffix));
	}
	
	/**
	* Generate date form
	*
	* @param integer $year Year to select
	* @param integer $month Month to select
	* @param integer $day Day to select
	* @return array $tab html data with 3 cells :['days'], ['months'], ['years']
	*
	*/	
	static public function dateYears()
	{
		for ($i = date('Y') - 10; $i >= 1900; $i--)
			$tab[] = $i;
		return $tab;
	}
	
	static public function dateDays()
	{
		for ($i = 1; $i != 32; $i++)
			$tab[] = $i;
		return $tab;
	}
	
	static public function dateMonths()
	{
		for ($i = 1; $i != 13; $i++)
			$tab[$i] = date('F', mktime(0, 0, 0, $i, date('m'), date('Y')));
		return $tab;
	}

	static public function hourGenerate($hours, $minutes, $seconds)
	{
	    return implode(':', array($hours, $minutes, $seconds));
	}

	static public function dateFrom($date)
	{
		$tab = explode(' ', $date);
		if (!isset($tab[1]))
		    $date .= ' ' . Tools::hourGenerate(0, 0, 0);
		return $date;
	}

	static public function dateTo($date)
	{
		$tab = explode(' ', $date);
		if (!isset($tab[1]))
		    $date .= ' ' . Tools::hourGenerate(23, 59, 59);
		return $date;
	}
	
	static public function getExactTime() {
		return time()+microtime();
	}
}

?>
